/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.generator.writer;

import static com.google.common.collect.Streams.concat;
import static com.sap.cds.generator.util.NamesUtils.checkForJavaKeyword;
import static com.sap.cds.generator.util.NamesUtils.getResolvedWrapperName;
import static com.sap.cds.generator.util.NamesUtils.isTechnicalEntity;
import static com.sap.cds.generator.util.NamesUtils.packageName;
import static com.sap.cds.generator.util.NamesUtils.qualifiedContextname;
import static com.sap.cds.generator.util.NamesUtils.qualifiedWrapperBuilderName;
import static com.sap.cds.generator.util.NamesUtils.unqualifiedName;
import static com.sap.cds.generator.util.TypeUtils.builderClassName;
import static com.sap.cds.generator.util.TypeUtils.className;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.lang.model.element.Modifier;
import javax.tools.JavaFileObject;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.util.EntityWriterException;
import com.sap.cds.generator.util.GeneratedFile;
import com.sap.cds.generator.util.GeneratedFile.Consumer;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsAnnotation;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsEvent;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.TypeSpec;

public class ModelWriter implements CdsVisitor {

	private static final String CONTEXT_SUFFIX = "Context";
	private final Consumer consumer;
	private final Configuration config;
	private final CdsModel model;
	private final Map<String, CdsDefinition> wrapperMap;
	private final Set<String> builderInterfaces;
	private final boolean isEventContext;
	private final NamesUtils namesUtils;

	public ModelWriter(Consumer consumer, Configuration config, CdsModel model) {
		this.consumer = consumer;
		this.config = config;
		this.model = model;
		this.wrapperMap = new HashMap<>();
		this.builderInterfaces = new HashSet<>();
		this.isEventContext = Boolean.parseBoolean(config.getEventContext());
		this.namesUtils = new NamesUtils(config.getBasePackage());
	}

	@Override
	public void visit(CdsModel model) {
		wrapperMap.values().forEach(this::generateWrapperInterface);
	}

	@Override
	public void visit(CdsEntity entity) {
		checkForJavaKeyword(entity.getQualifiedName());
		if (!isExcluded(entity.getQualifiedName()) && !isTechnicalEntity(model, entity.getQualifiedName())) {
			collectWrapperInterfaces(entity);
			generateBuilderInterface(entity);
			generateConsumptionInterface(entity);
			if (isEventContext) {
				concat(entity.actions(), entity.functions()).forEach(a -> generateEventContextInterface(a, entity));
			}
		}
	}

	@Override
	public void visit(CdsEvent event) {
		checkForJavaKeyword(event.getQualifiedName());
		if (!isExcluded(event.getQualifiedName())) {
			collectWrapperInterfaces(event);
			generateConsumptionInterface(event);
			if (isEventContext) {
				generateEventContextInterface(event, null);
			}
		}
	}

	@Override
	public void visit(CdsAction action) {
		if (isEventContext && !isExcluded(action.getQualifiedName())) {
			collectWrapperInterfaces(action);
			generateEventContextInterface(action, null);
		}
	}

	@Override
	public void visit(CdsFunction function) {
		if (isEventContext && !isExcluded(function.getQualifiedName())) {
			collectWrapperInterfaces(function);
			generateEventContextInterface(function, null);
		}
	}

	@Override
	public void visit(CdsStructuredType struct) {
		if (!struct.getName().isEmpty()) {
			checkForJavaKeyword(struct.getQualifiedName());
			if (!isExcluded(struct.getQualifiedName())) {
				generateConsumptionInterface(struct);
				if (isTargetAspect(struct)) {
					generateBuilderInterface(struct);
				}
			}
		}
	}

	private void collectWrapperInterfaces(CdsDefinition def) {
		String packageName = namesUtils.packageName(def);
		wrapperMap.put(packageName, def);
	}

	private void generateWrapperInterface(CdsDefinition def) {
		String qualifiedName = def.getQualifiedName();
		String name = def.getName();
		String packageName = namesUtils.packageName(def);
		String qualifiedWrapperName = qualifiedWrapperBuilderName(def, config.getClassNameSuffix(), true);
		while (builderInterfaces.contains(qualifiedWrapperName.toLowerCase(Locale.US))) {
			qualifiedWrapperName = getResolvedWrapperName(qualifiedWrapperName, config.getClassNameSuffix());
		}

		TypeSpec.Builder builder = TypeSpec.interfaceBuilder(unqualifiedName(qualifiedWrapperName))
				.addModifiers(Modifier.PUBLIC);
		if (WrapperInterfaceCreator.create(builder, model, config)
				.generateInterface(qualifiedContextname(qualifiedName, name))) {
			writeType(packageName, builder.build());
		}
	}

	private void generateConsumptionInterface(CdsDefinition def) {
		TypeSpec.Builder builder = createInterfaceSpecBuilder(className(def.getName()));
		String qualifiedTypeName = namesUtils.qualifiedJavaClassName(def);
		def.accept(CreateConsumptionInterfaceVisitor.create(builder, config, qualifiedTypeName));

		Optional<CdsAnnotation<Object>> parentAnnotation = def
				.findAnnotation(CdsConstants.ANNOTATION_JAVA_EXTENDS.substring(1));
		parentAnnotation.ifPresent(annotation -> {
			@SuppressWarnings("unchecked")
			ArrayList<String> baseEntities = (ArrayList<String>) annotation.getValue();
			for (String baseEntityName : baseEntities) {
				CdsEntity baseEntity = model.getEntity(baseEntityName);
				String packageNameForBase = packageName(config.getBasePackage(), baseEntity.getQualifiedName());
				builder.addSuperinterface(ClassName.get(packageNameForBase, baseEntity.getName()));
			}
		});

		TypeSpec typeSpec = builder.build();
		String packageName = namesUtils.packageName(def);
		writeType(packageName, typeSpec);
	}

	private void generateEventContextInterface(CdsDefinition def, CdsDefinition boundEntity) {
		TypeSpec.Builder builder = createInterfaceSpecBuilder(className(def.getName() + CONTEXT_SUFFIX));
		String boundEntityName = boundEntity == null ? null : boundEntity.getQualifiedName();

		def.accept(CreateEventContextInterfaceVisitor.create(builder, config, boundEntityName, def.getQualifiedName(),
				namesUtils));

		TypeSpec typeSpec = builder.build();
		String packageName = boundEntity != null ? namesUtils.packageName(boundEntity) : namesUtils.packageName(def);
		writeType(packageName, typeSpec);
	}

	private void generateBuilderInterface(CdsDefinition def) {
		TypeSpec.Builder builder = createInterfaceSpecBuilder(builderClassName(def));

		def.accept(CreateBuilderInterfaceVisitor.create(builder, config, namesUtils));

		builderInterfaces
				.add(qualifiedWrapperBuilderName(def, config.getClassNameSuffix(), false).toLowerCase(Locale.US));
		TypeSpec typeSpec = builder.build();
		String packageName = namesUtils.packageName(def);
		writeType(packageName, typeSpec);
	}

	private static TypeSpec.Builder createInterfaceSpecBuilder(ClassName type) {
		return TypeSpec.interfaceBuilder(type).addModifiers(Modifier.PUBLIC);
	}

	private boolean isExcluded(String qualifiedName) {
		List<String> excludes = config.getExcludes();
		boolean excluded = false;
		int i = qualifiedName.lastIndexOf('.');
		if (i > 0) {
			String namespace = qualifiedName.substring(0, i);
			excluded = excludes.stream().filter(e -> !e.contains("*")).anyMatch(e -> e.equals(namespace));
		}
		if (!excluded) {
			excluded = excludes.stream().filter(e -> e.contains("*")).anyMatch(qualifiedName::matches);
		}
		return excluded;
	}

	private boolean isTargetAspect(CdsStructuredType struct) {
		return model.entities().flatMap(CdsStructuredType::associations)
				.map(a -> a.getType().as(CdsAssociationType.class).getTargetAspect())
				.anyMatch(ta -> ta.isPresent() && ta.get().getName().equals(struct.getName()));
	}

	private void writeType(String packageName, TypeSpec typeSpec) {
		final JavaFile javaFile = JavaFile.builder(packageName, typeSpec).build();
		final JavaFileObject fileObject = javaFile.toJavaFileObject();
		GeneratedFile jpaFile = new GeneratedFile() {
			@Override
			public URI getUri() {
				return fileObject.toUri();
			}

			@Override
			public InputStream getContent() throws IOException {
				return fileObject.openInputStream();
			}
		};
		try {
			consumer.accept(jpaFile);
		} catch (IOException e) {
			String message = String.format("Exception while writing to file %s.", jpaFile.getUri());
			throw new EntityWriterException(message, e);
		}
	}
}
