package com.sap.cds.generator.util;

import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;

import java.util.Locale;
import java.util.Optional;
import java.util.Set;

import javax.lang.model.SourceVersion;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;

public class NamesUtils {

	private static final String UP = "up_";
	private static final String DOT = ".";
	private static final Logger logger = LoggerFactory.getLogger(NamesUtils.class);
	private final String basePackage;

	public NamesUtils(String basePackage) {
		this.basePackage = basePackage;
	}

	/**
	 * <p>
	 * Return package based on basePackage and qualifiedName.<br>
	 * Example:<br>
	 * a) basepackage = EMPTY; OUT = model<br>
	 * b) basepackage = EMPTY; qualifiedName = com.Model; OUT = com<br>
	 * c) basepackage = com, qualifiedName = Model; OUT = com<br>
	 * d) basepackage = com, qualifiedName = sap.Model; OUT = com.sap
	 * </p>
	 * 
	 * @param basePackage   the default package name prefix
	 * @param qualifiedName the fully qualified entity name
	 * @return String computed lowercase package name
	 * 
	 */
	public static String packageName(String basePackage, String qualifiedName) {
		String packageName = namespace(qualifiedName);
		return getPackageName(basePackage, packageName);
	}

	private static String getPackageName(String basePackage, String packageName) {
		if (!Strings.isNullOrEmpty(basePackage)) {
			packageName = Joiner.on('.').skipNulls().join(basePackage, packageName);
		}
		if (Strings.isNullOrEmpty(packageName)) {
			packageName = "model";
		}
		return packageName.toLowerCase(Locale.US);
	}

	public static String namespace(String qualifiedName) {
		int lastDot = qualifiedName.lastIndexOf('.');
		if (lastDot != -1) {
			return qualifiedName.substring(0, lastDot);
		}
		return null;
	}

	public static String qualifiedWrapperBuilderName(CdsDefinition def, String classNameSuffix, boolean isWrapper) {
		String name = (isWrapper ? toUpperCamel(unqualifiedContextName(def.getQualifiedName(), def.getName()))
				: toUpperCamel(def.getName())) + classNameSuffix;
		if (def.getQualifiedName().equals(def.getName())) {// Wrapper name is 'CdsModel_'
			return name;
		}
		return qualifiedContextname(def.getQualifiedName(), def.getName()) + "." + name;
	}

	public static String qualifiedContextname(String qualifiedName, String name) {
		if (name.contains(DOT)) {
			return getQualifiedContextNameForDot(qualifiedName, name);
		} else {
			int lastDot = qualifiedName.lastIndexOf('.');
			if (lastDot != -1) {
				return qualifiedName.substring(0, lastDot);
			}
			return null;
		}
	}

	public static String unqualifiedContextName(String qualifiedName, String name) {
		if (name.contains(DOT)) {
			String substring = getQualifiedContextNameForDot(qualifiedName, name);
			if (substring.contains(DOT)) {
				String[] bits = substring.split("\\.");
				return bits[bits.length - 1];
			}
			return substring;
		} else if (qualifiedName.contains(DOT)) {
			String[] bits = qualifiedName.split("\\.");
			return bits[bits.length - 2];
		}
		return "CdsModel";
	}

	public static String getQualifiedContextNameForDot(String qualifiedName, String name) {
		String qualifiedContextName = qualifiedName.substring(0, qualifiedName.lastIndexOf(name) - 1);
		if (Strings.isNullOrEmpty(qualifiedContextName)) {
			return "CdsModel";
		}
		return qualifiedContextName;
	}

	public static String unqualifiedName(String qualifiedName) {
		int lastDot = qualifiedName.lastIndexOf('.');

		return qualifiedName.substring(lastDot + 1);
	}

	public static String qualifiedJavaClass(String basePackage, String entityName) {
		String packageName = packageName(basePackage, entityName);
		String name = unqualifiedName(entityName);

		return packageName + DOT + toUpperCamel(name);
	}

	public static String qualifiedJavaClassName(String basePackage, String qualifiedName, String name) {
		String namespace = namespace(qualifiedName, name);
		String packageName = getPackageName(basePackage, namespace);

		return packageName + DOT + toUpperCamel(name);
	}

	public static void checkForJavaKeyword(String qualifiedName) {
		int lastDot = qualifiedName.lastIndexOf('.');
		if (lastDot != -1) {
			String[] split = qualifiedName.split("\\.");
			for (String word : split) {
				if (SourceVersion.isKeyword(word)) {
					logger.warn("The Entity " + qualifiedName
							+ " contains a reserved Java keyword in its fully qualified name.");
				}
			}
		}
	}

	public static String namespace(String qualifiedName, String name) {
		if (qualifiedName.equals(name)) {
			return null;
		} else {
			return qualifiedName.substring(0, qualifiedName.lastIndexOf(name) - 1);
		}
	}

	public static boolean isValidTechnicalEntity(CdsModel model, String qualifiedName, Set<String> entitiesWithAspectContainingStruct) {
		Optional<CdsEntity> findEntity = model.findEntity(qualifiedName);
		boolean isTechnicalEntity = false;
		if (findEntity.isPresent()) {
			isTechnicalEntity = findEntity.get().elements().anyMatch(e -> e.getName().startsWith(UP));
		}
		//Check if Entity contains composition over aspects containing structured element 
		//if so, this is a valid technical entity for which interfaces will be generated.
		return !isTechnicalEntity || entitiesWithAspectContainingStruct.contains(qualifiedName);
	}

	public static String innerInterfaceQualifiedName(CdsElement element, String basePackage) {
		String qualifiedElementName = element.getQualifiedName();
		int namespaceBeginIndex = qualifiedElementName.lastIndexOf(element.getDeclaringType().getName() + ":");
		String namespace = namespaceBeginIndex == 0 ? null
				: qualifiedElementName.substring(0, namespaceBeginIndex - 1).toLowerCase();
		String entityName = toUpperCamel(element.getDeclaringType().getName());
		if (!Strings.isNullOrEmpty(basePackage)) {
			return Joiner.on('.').skipNulls().join(basePackage, namespace, entityName, toUpperCamel(element.getName()));
		}
		return Joiner.on('.').skipNulls().join(namespace, entityName, toUpperCamel(element.getName()));
	}

	public static String getResolvedWrapperName(String qualifiedBuilderName, String classNameSuffix) {
		return qualifiedBuilderName.substring(0, qualifiedBuilderName.length() - 1) + "Model" + classNameSuffix;
	}

	public String packageName(CdsDefinition def) {
		String packageName = namespace(def.getQualifiedName(), def.getName());
		return getPackageName(basePackage, packageName);
	}

	public String qualifiedJavaClassName(CdsDefinition def) {
		String packageName = packageName(def);

		return packageName + DOT + toUpperCamel(def.getName());
	}

}
