/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.PoetTypeName.getTypeName;
import static com.sap.cds.generator.util.TypeUtils.builderClassName;
import static com.sap.cds.generator.util.TypeUtils.getAnonymousElements;
import static com.sap.cds.generator.util.TypeUtils.getReturnType;
import static com.sap.cds.generator.util.TypeUtils.isCompositionOfAspectsWithStructuredElements;
import static com.sap.cds.generator.util.TypeUtils.isAnonymousAspect;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;
import static com.sap.cds.generator.writer.SpecWriterUtil.addCdsNameAnnotation;
import static com.sap.cds.generator.writer.SpecWriterUtil.addStaticField;
import static com.sap.cds.reflect.impl.CdsAnnotatableImpl.removeAt;
import static com.sap.cds.reflect.impl.reader.model.CdsConstants.ANNOTATION_CDS_JAVA_NAME;
import static javax.lang.model.element.Modifier.ABSTRACT;
import static javax.lang.model.element.Modifier.PUBLIC;

import java.util.function.Function;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.generator.util.PoetTypeName;
import com.sap.cds.ql.CdsName;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.CdsVisitor;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class CreateBuilderInterfaceVisitor implements CdsVisitor {

	private static final Logger logger = LoggerFactory.getLogger(CreateBuilderInterfaceVisitor.class);
	private static final String FILTER = "filter";
	private final TypeSpec.Builder entityClass;
	private final Configuration cfg;
	private final NamesUtils namesUtils;
	private final boolean containsEntityStructAspect;
	private String entityName;

	private CreateBuilderInterfaceVisitor(TypeSpec.Builder builder, Configuration cfg, NamesUtils namesUtils,
			boolean containsEntityStructAspect) {
		this.entityClass = builder;
		this.namesUtils = namesUtils;
		this.cfg = cfg;
		this.containsEntityStructAspect = containsEntityStructAspect;
	}

	public static CdsVisitor create(TypeSpec.Builder builder, Configuration cfg, NamesUtils namesUtils,
			boolean containsEntityStructAspect) {
		return new CreateBuilderInterfaceVisitor(builder, cfg, namesUtils, containsEntityStructAspect);
	}

	@Override
	public void visit(CdsEntity entity) {
		generateInterface(entity);
	}

	@Override
	public void visit(CdsStructuredType struct) {
		generateInterface(struct);
	}

	@Override
	public void visit(CdsElement attribute) {
		if (attribute.getType().isStructured() && attribute.getType().as(CdsStructuredType.class).isAnonymous()) {
			addAnonymousStructAttribute(attribute);
		} else if (!attribute.getType().isAssociation()) {
			addAttribute(attribute);
		} else {
			addAssociationAttribute(attribute);
		}
	}

	private void generateInterface(CdsDefinition def) {
		TypeName type = builderClassName(def);
		
		entityClass.addSuperinterface(ParameterizedTypeName.get(Types.STRUCTUREDTYPE, type));
		entityClass.addAnnotation(
				AnnotationSpec.builder(CdsName.class).addMember("value", "$S", def.getQualifiedName()).build());
		addStaticQualifiedAttribute(def);
		entityName = def.getName();
	}
	
	private void addAnonymousStructAttribute(CdsElement attribute) {
		String unqualifiedClassName = toUpperCamel(attribute.getName()) + cfg.getClassNameSuffix();
		ParameterizedTypeName returnType = ParameterizedTypeName.get(Types.REFERENCE,
				ClassName.bestGuess(unqualifiedClassName));
		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(attribute.getName())
				.addModifiers(PUBLIC, ABSTRACT).returns(returnType);
		entityClass.addMethod(methodBuilder.build());

		//generate inner interface
		Stream<CdsElement> elements = attribute.getType().as(CdsStructuredType.class).elements();
		addInnerInterface(attribute, unqualifiedClassName, elements);
	}

	private void addInnerInterface(CdsElement attribute, String unqualifiedClassName, Stream<CdsElement> elements) {
		TypeSpec.Builder innerIntefaceBuilder = TypeSpec.interfaceBuilder(unqualifiedClassName)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);
		innerIntefaceBuilder.addSuperinterface(
				ParameterizedTypeName.get(Types.STRUCTUREDTYPE, PoetTypeName.getTypeName(unqualifiedClassName)));
		elements.forEach(
				e -> e.accept(CreateBuilderInterfaceVisitor.create(innerIntefaceBuilder, cfg, namesUtils, false)));
		entityClass.addType(innerIntefaceBuilder.build());
	}

	private void addStaticQualifiedAttribute(CdsDefinition def) {
		String qualifiedName = def.getQualifiedName();

		FieldSpec staticField = FieldSpec.builder(String.class, "CDS_NAME")
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL).initializer("$S", qualifiedName)
				.build();
		entityClass.addField(staticField);
	}

	private void addAssociationAttribute(CdsElement attribute) {
		if (isAnonymousAspect(attribute)) {
			String unqualifiedClassName = (isCompositionOfAspectsWithStructuredElements(attribute)
					? toUpperCamel(attribute.getType().as(CdsAssociationType.class).getTarget().getName())
					: toUpperCamel(attribute.getName())) + cfg.getClassNameSuffix();
			addInnerInterface(attribute, unqualifiedClassName, getAnonymousElements(attribute));
			addMainFunction(attribute);
			addFilterFunction(attribute);
		} else {
			addMainFunction(attribute);
			addFilterFunction(attribute);
		}
	}

	private void addAttribute(CdsElement attribute) {
		String attributeName = attribute.getAnnotationValue(removeAt(ANNOTATION_CDS_JAVA_NAME), attribute.getName());
		ParameterizedTypeName returnType = ParameterizedTypeName.get(Types.REFERENCE, getReturnType(attribute, cfg));
		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(attributeName).addModifiers(PUBLIC, ABSTRACT)
				.returns(returnType);
		if (addCdsNameAnnotation(methodBuilder, attribute.getName(), attributeName)) {
			addStaticField(entityClass, attribute);
		}
		entityClass.addMethod(methodBuilder.build());
	}

	private void addMainFunction(CdsElement attribute) {
		String targetEntityClassName = getTargetName(attribute);
		TypeName returnType = PoetTypeName.getTypeName(targetEntityClassName);
		entityClass.addMethod(MethodSpec.methodBuilder(attribute.getName()).addModifiers(PUBLIC, ABSTRACT)
				.returns(returnType).build());

	}

	private void addFilterFunction(CdsElement attribute) {
		if (!attribute.getName().equalsIgnoreCase(FILTER)) {
			String targetEntityClassName = getTargetName(attribute);
			TypeName returnType = PoetTypeName.getTypeName(targetEntityClassName);
			entityClass
					.addMethod(MethodSpec.methodBuilder(attribute.getName())
							.addParameter(ParameterizedTypeName.get(ClassName.get(Function.class), returnType,
									Types.PREDICATE), "filter")
							.addModifiers(PUBLIC, ABSTRACT).returns(returnType).build());
		} else {
			logger.warn("There is a name clash between the element 'filter' of entity '" + entityName
					+ "' and a method in the super interface StructuredType. "
					+ "A filter function accepting predicates will not be generated.");
		}
	}

	public String getTargetName(CdsElement element) {
		CdsAssociationType association = element.getType();
		CdsType target;
		if (containsEntityStructAspect && isCompositionOfAspectsWithStructuredElements(element)) {
			if (!isAnonymousAspect(element)) {
				return interfaceName(association.getTarget());
			} else {
				return toUpperCamel(association.getTarget().getName()) + cfg.getClassNameSuffix();
			}
		} else {
			target = association.getTargetAspect().orElseGet(association::getTarget);
		}
		if (target.getQualifiedName().isEmpty()) {
			return toUpperCamel(element.getName()) + cfg.getClassNameSuffix();
		}
		return interfaceName(target);
	}

	public String interfaceName(CdsDefinition def) {
		return getTypeName(namesUtils.qualifiedJavaClassName(def)) + cfg.getClassNameSuffix();
	}
}
