/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.NamesUtils.getQualifiedContextNameForDot;
import static com.sap.cds.generator.util.NamesUtils.isValidTechnicalEntity;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperUnderscore;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.generator.util.PoetTypeName;
import com.sap.cds.ql.CdsName;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class WrapperInterfaceCreator {

	private static final String DOT = ".";
	private final TypeSpec.Builder builder;
	private final CdsModel model;
	private final Configuration cfg;
	private final NamesUtils namesUtils;
	private final Set<String> entitiesWithAspectContainingStruct;

	public static WrapperInterfaceCreator create(TypeSpec.Builder builder, CdsModel model, Configuration cfg, Set<String> entitiesWithAspectContainingStruct) {
		return new WrapperInterfaceCreator(builder, model, cfg, entitiesWithAspectContainingStruct);
	}

	private WrapperInterfaceCreator(TypeSpec.Builder builder, CdsModel model, Configuration cfg, Set<String> entitiesWithAspectContainingStruct) {
		this.builder = builder;
		this.model = model;
		this.cfg = cfg;
		this.namesUtils = new NamesUtils(cfg);
		this.entitiesWithAspectContainingStruct = entitiesWithAspectContainingStruct;
	}

	public boolean generateInterface(String context) {
		List<CdsDefinition> definitionList = getEntities(context);
		definitionList.addAll(getEvents(context));
		definitionList.addAll(getActionsAndFunctions(context));
		if (definitionList.isEmpty()) {
			return false;
		}
		if (context != null) {
			builder.addAnnotation(AnnotationSpec.builder(CdsName.class).addMember("value", "$S", context).build());
			addStaticQualifiedAttribute(context);
		}

		definitionList.stream().forEach(d -> {
			if (d instanceof CdsEntity) {
				TypeName type;
				String interfaceName = toUpperCamel(d.getName()) + cfg.getClassNameSuffix();
				type = ParameterizedTypeName.get(ClassName.get(Class.class), PoetTypeName.getTypeName(interfaceName));
				FieldSpec staticField = FieldSpec.builder(type, toUpperUnderscore(toUpperCamel(d.getName())))
						.addModifiers(Modifier.PUBLIC, Modifier.FINAL, Modifier.STATIC)
						.initializer("$N.class", interfaceName).build();
				builder.addField(staticField);
			}
		});
		return true;
	}

	private List<CdsDefinition> getEvents(String context) {
		if (context == null) {
			return model.events().filter(e -> !e.getQualifiedName().contains(DOT)).collect(Collectors.toList());
		}
		return model.events().filter(e -> filterDefinitions(context, e) && !namesUtils.isExcluded(e.getQualifiedName())).collect(Collectors.toList());
	}

	private List<CdsDefinition> getActionsAndFunctions(String context) {
		List<CdsDefinition> definitionList = new ArrayList<CdsDefinition>();
		if (context == null) {
			model.actions().filter(a -> !a.getQualifiedName().contains(DOT)).forEach(definitionList::add);
			model.functions().filter(f -> !f.getQualifiedName().contains(DOT)).forEach(definitionList::add);
		}
		model.actions().filter(a -> filterDefinitions(context, a) && !namesUtils.isExcluded(a.getQualifiedName())).forEach(definitionList::add);
		model.functions().filter(f -> filterDefinitions(context, f) && !namesUtils.isExcluded(f.getQualifiedName())).forEach(definitionList::add);

		return definitionList;
	}

	private void addStaticQualifiedAttribute(String context) {
		FieldSpec staticEntityField = FieldSpec.builder(String.class, "CDS_NAME")
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL).initializer("$S", context).build();
		builder.addField(staticEntityField);
	}

	private List<CdsDefinition> getEntities(String context) {
		if (context == null) {
			return model.concreteEntities()
					.filter(e -> !e.getQualifiedName().contains(".") && isValidTechnicalEntity(model, e.getQualifiedName(), entitiesWithAspectContainingStruct))
					.collect(Collectors.toList());
		}
		return model.concreteEntities()
				.filter(e -> filterDefinitions(context, e) && !namesUtils.isExcluded(e.getQualifiedName()) && isValidTechnicalEntity(model, e.getQualifiedName(), entitiesWithAspectContainingStruct))
				.collect(Collectors.toList());
	}

	private boolean filterDefinitions(String context, CdsDefinition def) {
		String name = def.getName();
		String qualifiedName = def.getQualifiedName();
		if (name.contains(DOT)) {
			String qualifiedContextName = getQualifiedContextNameForDot(qualifiedName, name);
			return qualifiedContextName.equalsIgnoreCase(context);
		}
		if (qualifiedName.contains(DOT)) {
			String contextPath = qualifiedName.substring(0, qualifiedName.lastIndexOf('.'));
			return contextPath.equalsIgnoreCase(context);
		}
		return false;
	}

}
