/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.generator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

import com.google.common.base.Strings;
import com.sap.cds.generator.util.GeneratorMode;
import com.sap.cds.generator.util.ParserMode;

/**
 * Configuration instance stores the configuration values for the application.
 * There should exist only one instance per application. Normally should be
 * filled from the command line options. But for the tests and integration one
 * can create an empty configuration and use default values.
 *
 */
public class ConfigurationImpl implements Configuration {

	private ParserMode parserMode = ParserMode.TOLERANT;
	private GeneratorMode generatorMode = GeneratorMode.TOLERANT;
	private MethodStyle methodStyle = MethodStyle.BEAN;
	private String classNameSuffix = "_";
	private String basePackage = "";
	private String eventContext = "";
	private List<String> excludes = new ArrayList<>();
	private List<String> includes = new ArrayList<>();
	private boolean docs = true;
	private boolean fkAccessors = false;
	private boolean generatedAnnotation;
	private String pluginInformation = "";

	@Override
	public GeneratorMode getGeneratorMode() {
		return generatorMode;
	}

	public void setGeneratorMode(GeneratorMode generatorMode) {
		this.generatorMode = generatorMode;
	}

	public void setClassNameSuffix(String suffix) {
		this.classNameSuffix = suffix;
	}

	@Override
	public String getClassNameSuffix() {
		return classNameSuffix;
	}

	public void setGeneratorMode(String generatorMode) {
		if (!Strings.isNullOrEmpty(generatorMode)) {
			this.generatorMode = GeneratorMode.valueOf(generatorMode.toUpperCase(Locale.US));
		}
	}

	@Override
	public ParserMode getParserMode() {
		return parserMode;
	}

	public void setParserMode(ParserMode parserMode) {
		this.parserMode = parserMode;
	}

	public void setParserMode(String parserMode) {
		if (!Strings.isNullOrEmpty(parserMode)) {
			this.parserMode = ParserMode.valueOf(parserMode.toUpperCase(Locale.US));
		}
	}

	@Override
	public MethodStyle getMethodStyle() {
		return methodStyle;
	}

	public void setMethodStyle(MethodStyle methodMode) {
		this.methodStyle = methodMode;
	}

	public void setMethodStyle(String interfaceMode) {
		if (!Strings.isNullOrEmpty(interfaceMode)) {
			this.methodStyle = MethodStyle.valueOf(interfaceMode.toUpperCase(Locale.US));
		}
	}

	@Override
	public String getBasePackage() {
		return this.basePackage;
	}

	public void setBasePackage(String basePackage) {
		if (!Strings.isNullOrEmpty(basePackage)) {
			this.basePackage = basePackage;
		}
	}

	@Override
	public List<String> getExcludes() {
		return Collections.unmodifiableList(excludes);
	}

	public void setExcludes(List<String> excludes) {
		if (excludes != null) {
			this.excludes.addAll(excludes);
		}
	}

	@Override
	public String getEventContext() {
		return this.eventContext;
	}

	public void setEventContext(String bool) {
		if (!Strings.isNullOrEmpty(bool)) {
			this.eventContext = bool;
		}
	}

	@Override
	public boolean getDocs() {
		return this.docs;
	}

	public void setDocs(boolean readDocs) {
		this.docs = readDocs;
	}

	@Override
	public List<String> getIncludes() {
		return Collections.unmodifiableList(includes);
	}

	public void setIncludes(List<String> includes) {
		if (includes != null) {
			this.includes.addAll(includes);
		}
	}

	@Override
	public boolean fkAccessors() {
		return this.fkAccessors;
	}

	public void setFkAccessors(boolean fkAccessors) {
		this.fkAccessors = fkAccessors;
	}

	@Override
	public void setGeneratedAnnotation(boolean generate) {
		this.generatedAnnotation = generate;
	}

	@Override
	public boolean getGeneratedAnnotation() {
		return this.generatedAnnotation;
	}

	@Override
	public String getPluginInformation() {
		return pluginInformation;
	}

	public void setPluginInformation(String pluginInformation) {
		this.pluginInformation = pluginInformation;
	}
}
