/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.util;

import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperUnderscore;
import static com.sap.cds.reflect.impl.reader.model.CdsConstants.ANNOTATION_CDS_JAVA_IGNORE;
import static com.sap.cds.reflect.impl.reader.model.CdsConstants.ANNOTATION_CDS_JAVA_NAME;
import static com.sap.cds.reflect.impl.reader.model.CdsConstants.ANNOTATION_ODATA_FK4;

import java.util.List;
import java.util.Locale;
import java.util.Optional;

import javax.lang.model.SourceVersion;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.sap.cds.generator.Configuration;
import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsService;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.NameAllocator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NamesUtils {

	private static final String CDS_MODEL = "CdsModel";
	private static final String UP = "up_";
	private static final String DOT = ".";
	private static final Logger logger = LoggerFactory.getLogger(NamesUtils.class);
	private final List<String> excludes;
	private final List<String> includes;
	private final String basePackage;
	public static final String ITEM_TYPE_NAME = "Item";

	public NamesUtils(Configuration config) {
		this.basePackage = config.getBasePackage();
		this.excludes = config.getExcludes();
		this.includes = config.getIncludes();
	}

	/**
	 * <p>
	 * Return package based on basePackage and qualifiedName.<br>
	 * Example:<br>
	 * a) basepackage = EMPTY; OUT = model<br>
	 * b) basepackage = EMPTY; qualifiedName = com.Model; OUT = com<br>
	 * c) basepackage = com, qualifiedName = Model; OUT = com<br>
	 * d) basepackage = com, qualifiedName = sap.Model; OUT = com.sap
	 * </p>
	 *
	 * @param basePackage   the default package name prefix
	 * @param qualifiedName the fully qualified entity name
	 * @return String computed lowercase package name
	 *
	 */
	public static String packageName(String basePackage, String qualifiedName) {
		String packageName = namespace(qualifiedName);
		return getPackageName(basePackage, packageName);
	}

	private static String getPackageName(String basePackage, String packageName) {
		if (!Strings.isNullOrEmpty(basePackage)) {
			packageName = Joiner.on('.').skipNulls().join(basePackage, packageName);
		}
		if (Strings.isNullOrEmpty(packageName)) {
			packageName = "model";
		}
		return packageName.toLowerCase(Locale.US);
	}

	public static String namespace(String qualifiedName) {
		int lastDot = qualifiedName.lastIndexOf('.');
		if (lastDot != -1) {
			return qualifiedName.substring(0, lastDot);
		}
		return null;
	}

	public static String qualifiedWrapperBuilderName(CdsDefinition def, String classNameSuffix, boolean isWrapper) {
		String name = (isWrapper ? toUpperCamel(unqualifiedContextName(def.getQualifiedName(), def.getName()))
				: toUpperCamel(def.getName())) + classNameSuffix;
		if (def.getQualifiedName().equals(def.getName())) {// Wrapper name is 'CdsModel_'
			return name;
		}
		return qualifiedContextname(def.getQualifiedName(), def.getName()) + "." + name;
	}

	public static String qualifiedContextname(String qualifiedName, String name) {
		if (name.contains(DOT)) {
			return getQualifiedContextNameForDot(qualifiedName, name);
		} else {
			int lastDot = qualifiedName.lastIndexOf('.');
			if (lastDot != -1) {
				return qualifiedName.substring(0, lastDot);
			}
			return CDS_MODEL;
		}
	}

	public static String unqualifiedContextName(String qualifiedName, String name) {
		if (name.contains(DOT)) {
			String substring = getQualifiedContextNameForDot(qualifiedName, name);
			if (substring.contains(DOT)) {
				String[] bits = substring.split("\\.");
				return bits[bits.length - 1];
			}
			return substring;
		} else if (qualifiedName.contains(DOT)) {
			String[] bits = qualifiedName.split("\\.");
			return bits[bits.length - 2];
		}
		return CDS_MODEL;
	}

	public static String getQualifiedContextNameForDot(String qualifiedName, String name) {
		String qualifiedContextName = null;
		int endIndex = qualifiedName.lastIndexOf(name) - 1;
		if(endIndex > 0) {
			qualifiedContextName = qualifiedName.substring(0, endIndex);
		}
		if (Strings.isNullOrEmpty(qualifiedContextName)) {
			return CDS_MODEL;
		}
		return qualifiedContextName;
	}

	public static String unqualifiedName(String qualifiedName) {
		int lastDot = qualifiedName.lastIndexOf('.');

		return qualifiedName.substring(lastDot + 1);
	}

	public static String qualifiedJavaClass(String basePackage, String entityName) {
		String packageName = packageName(basePackage, entityName);
		String name = unqualifiedName(entityName);

		return packageName + DOT + toUpperCamel(name);
	}

	public static String qualifiedJavaClassName(String basePackage, String qualifiedName, String name) {
		String namespace = namespace(qualifiedName, name);
		String packageName = getPackageName(basePackage, namespace);

		return packageName + DOT + toUpperCamel(name);
	}

	public static void checkForJavaKeyword(String qualifiedName) {
		int lastDot = qualifiedName.lastIndexOf('.');
		if (lastDot != -1) {
			String[] split = qualifiedName.split("\\.");
			for (String word : split) {
				if (SourceVersion.isKeyword(word)) {
					logger.warn("The Entity {} contains a reserved Java keyword in its fully qualified name.",
							qualifiedName);
				}
			}
		}
	}

	public static String namespace(String qualifiedName, String name) {
		if (qualifiedName.equals(name)) {
			return null;
		} else {
			return qualifiedName.substring(0, qualifiedName.lastIndexOf(name) - 1);
		}
	}

	public static boolean isValidTechnicalEntity(CdsModel model, String qualifiedName) {
		Optional<CdsEntity> findEntity = model.findEntity(qualifiedName);
		boolean isTechnicalEntity = false;
		if (findEntity.isPresent()) {
			isTechnicalEntity = findEntity.get().elements().anyMatch(e -> e.getName().startsWith(UP) && e.getType().isAssociation() && e.isKey());
		}
		return !isTechnicalEntity;
	}

	public static String innerInterfaceQualifiedName(CdsElement element, String basePackage) {
		String qualifiedElementName = element.getQualifiedName();
		int namespaceBeginIndex = qualifiedElementName.lastIndexOf(element.getDeclaringType().getName() + ":");
		String namespace = namespaceBeginIndex == 0 ? null
				: qualifiedElementName.substring(0, namespaceBeginIndex - 1).toLowerCase();
		String entityName = toUpperCamel(element.getDeclaringType().getName());
		if (!Strings.isNullOrEmpty(basePackage)) {
			return Joiner.on('.').skipNulls().join(basePackage, namespace, entityName, toUpperCamel(element.getName()));
		}
		return Joiner.on('.').skipNulls().join(namespace, entityName.length() == 0 ? null : entityName,
				toUpperCamel(element.getName()));
	}

	public static String getResolvedWrapperName(String qualifiedBuilderName, String classNameSuffix) {
		return qualifiedBuilderName.substring(0, qualifiedBuilderName.length() - 1) + "Model" + classNameSuffix;
	}

	/**
	 * Returns the effective package name of the given {@link CdsDefinition definition}.
	 *
	 * @param def         the {@link CdsDefinition}
	 * @param basePackage the base package
	 * @return the effective package name
	 */
	public static String packageName(CdsDefinition def, String basePackage) {
		String packageName = namespace(def.getQualifiedName(), def.getName());
		return getPackageName(basePackage, packageName);
	}

	/**
	 * Returns the effective package name of the given {@link CdsService service} definition.
	 *
	 * @param service     the {@link CdsService}
	 * @param basePackage the base package
	 * @return the effective package name
	 */
	public static String packageName(CdsService service, String basePackage) {
		return getPackageName(basePackage, service.getQualifiedName());
	}

	public String qualifiedJavaClassName(CdsDefinition def) {
		String packageName = packageName(def, basePackage);

		return packageName + DOT + toUpperCamel(def.getName());
	}

	public String validJavaMethodName(String name) {
		return name.replace('.', '_');
	}

	public boolean isExcluded(String qualifiedName) {
		boolean included = includes.isEmpty() || matchesAny(qualifiedName, includes);
		included &= !matchesAny(qualifiedName, excludes);
		return !included;
	}

	private boolean matchesAny(String qualifiedName, List<String> patternList) {
		return patternList.stream().map(p -> PatternMatcher.create(p)).anyMatch(pm -> pm.matches(qualifiedName));
	}

    /**
     * Returns the effective name of the given annotatable, if not annotated with
     * `@cds.java.ignore : true`
     *
     * @param annotatable a cds annotatable
     * @param defaultName the default name of the annotatable
     * @return the effective name of the cds element, or an empty optional if the
     *         element is to be ignored by code gen
     */
	public static Optional<String> getNameIfNotIgnored(CdsAnnotatable annotatable, String defaultName) {
		// FKs are technical elements, thus ignore further annotations
		if (annotatable.findAnnotation(ANNOTATION_ODATA_FK4).isPresent()) {
			return Optional.of(defaultName);
		}

		// element annotated with @cds.java.ignore (: true)
		if (annotatable.getAnnotationValue(ANNOTATION_CDS_JAVA_IGNORE, false)) {
			return Optional.empty();
		}

		return Optional.of(annotatable.getAnnotationValue(ANNOTATION_CDS_JAVA_NAME, defaultName));
	}

	public static String javaSafeConstantName(String name)	{
		if (SourceVersion.isName(name)) {
			return toUpperUnderscore(name);
		}
		return "_" + toUpperUnderscore(NameAllocator.toJavaIdentifier(name));
	}

	public static ClassName asNameWithItemsPostfix(String name) {
		// Generic PoetTypeName will always treat the namespace in the type as a package -> we need a name
		// WrapperInterface.Item for the global arrayed type with inline structure
		return ClassName.bestGuess(name + DOT + ITEM_TYPE_NAME);
	}
}
