/************************************************************************
 * © 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.writer.CaseFormatHelper.lowercaseFirst;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperUnderscore;
import static com.sap.cds.reflect.impl.CdsAnnotatableImpl.removeAt;
import static com.sap.cds.reflect.impl.reader.model.CdsConstants.ANNOTATION_CDS_JAVA_NAME;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.CdsName;
import com.sap.cds.ql.CdsPath;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.impl.CdsAnnotatableImpl;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeSpec;

public class SpecWriterUtil {

	public static boolean addCdsNameAnnotation(MethodSpec.Builder builder, CdsElement attribute, String javaName) {
		String attributeName = attribute.getAnnotationValue(removeAt(ANNOTATION_CDS_JAVA_NAME), attribute.getName());
		boolean addCdsNameAnnotation = !attribute.getName().equals(propertyName(javaName));
		if (addCdsNameAnnotation) {
			builder.addAnnotation(cdsNameAnnotation(toUpperUnderscore(attributeName), "$L"));
		}
		return addCdsNameAnnotation;
	}

	public static boolean addCdsPathAnnotation(MethodSpec.Builder builder, String cdsPath, String javaName) {
		boolean addCdsPathAnnotation = !cdsPath.equals(propertyName(javaName));
		if (addCdsPathAnnotation) {
			builder.addAnnotation(cdsPathAnnotation(toUpperUnderscore(cdsPath), "$L"));
		}
		return addCdsPathAnnotation;
	}

	public static AnnotationSpec cdsNameAnnotation(String cdsName, String format) {
		return AnnotationSpec.builder(CdsName.class).addMember("value", format, cdsName).build();
	}

	public static AnnotationSpec cdsPathAnnotation(String cdsName, String format) {
		return AnnotationSpec.builder(CdsPath.class).addMember("value", format, cdsName).build();
	}

	public static void addStaticField(TypeSpec.Builder builder, CdsElement element) {
		Optional<String> elementName = NamesUtils.getNameIfNotIgnored(element, element.getName());
		elementName.map(CaseFormatHelper::toUpperUnderscore).ifPresent(name -> {
			FieldSpec staticField = FieldSpec.builder(String.class, name)
			.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL).initializer("$S", element.getName())
			.build();
			builder.addField(staticField);
		});
	}

	public static Optional<String> getJavaDoc(CdsAnnotatable def) {
		return ((CdsAnnotatableImpl) def).getDoc();
	}

	public static void setJavaDoc(CdsAnnotatable attribute, MethodSpec.Builder methodBuilder) {
		getJavaDoc(attribute).ifPresent(a -> methodBuilder.addJavadoc(a.replace("$", "$$")));
	}

	public static void addFkStaticField(TypeSpec.Builder builder, CdsElement attribute) {
		// Use value refs from getFkMapping method to construct the path to foreign key
		List<String> keys = resolveKeys(attribute)
				.map(r -> r.path())
				.collect(Collectors.toList());
		
		if (!keys.isEmpty()) {
			keys.forEach(fk -> {
				FieldSpec field = FieldSpec.builder(String.class, toUpperUnderscore(fk.replace(".", "_")))
						.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL)
						.initializer("$S", fk).build();
				builder.addField(field);
			});
		}
	}
	
	private static Stream<CqnReference> resolveKeys(CdsElement assoc) {
		return resolveKeys(CQL.get(assoc.getName()), assoc.getType());
	}

	private static Stream<CqnReference> resolveKeys(CqnReference path, CdsAssociationType assoc) {
		return assoc.refs().flatMap(r -> {
			CqnReference ref = CQL.get(path.path() + "." + r.path());
			CdsType type = assoc.getTarget().getElement(r.path()).getType();
			if (type instanceof CdsAssociationType a) {
				return resolveKeys(ref, a);
			}
			return Stream.of(ref);
		});
	}

	private static String propertyName(String javaName) {
		if (javaName.startsWith("set") || javaName.startsWith("get")) {
			javaName = lowercaseFirst(javaName.substring(3));
		}
		return javaName;
	}

}
