/************************************************************************
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.util;

import static com.sap.cds.generator.util.NamesUtils.asNameWithItemsPostfix;
import static com.sap.cds.generator.util.NamesUtils.namespace;
import static com.sap.cds.generator.util.NamesUtils.qualifiedJavaClass;
import static com.sap.cds.generator.util.NamesUtils.qualifiedJavaClassName;
import static com.sap.cds.generator.util.NamesUtils.unqualifiedName;
import static com.sap.cds.generator.util.PoetTypeName.getArrayTypeName;
import static com.sap.cds.generator.util.PoetTypeName.getTypeFromCdsName;
import static com.sap.cds.generator.util.PoetTypeName.getTypeName;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;
import static com.sap.cds.generator.writer.Types.RETURN_TYPE;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.writer.Types;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsOperation;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.OnConditionAnalyzer;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TypeUtils {

	private static final Logger logger = LoggerFactory.getLogger(TypeUtils.class);

	private TypeUtils() {
	}

	public static TypeName getAttributeType(CdsType type, Configuration cfg) {
		return getAttributeType(type, cfg, null, null);
	}

	private static TypeName getAttributeType(CdsType type, Configuration cfg, String structTypeName, CdsElement element) {
		TypeName attributeType;

		if (type.isAssociation()) {
			CdsAssociationType assocType = type.as(CdsAssociationType.class);
			CdsType target = assocType.getTargetAspect().orElseGet(assocType::getTarget);
			attributeType = getTypeFromCdsName(cfg.getBasePackage(), target.getQualifiedName(), target.getName());
			if (!CdsModelUtils.isSingleValued(type)) {
				attributeType = listOf(attributeType);
			}
		} else if (type.isSimple()) {
			attributeType = getTypeName(type.as(CdsSimpleType.class).getJavaType().getTypeName());
		} else if (type.isStructured()) {
			String targetName = type.getQualifiedName();

			// arrayed inlined structured type
			if (targetName.isEmpty() && type.getName().isEmpty() && structTypeName != null) {
				attributeType = getTypeName(toUpperCamel(structTypeName));
			} else {
				attributeType = getTypeFromCdsName(cfg.getBasePackage(), targetName, type.getName());
			}
		} else if (type.isArrayed()) {
			// Two ways to fetch the type:
			if (element != null) {
				// inline types of an elements have names including the element name at the end
				attributeType = arrayedTypeNameForElement(type.as(CdsArrayedType.class), cfg, element);
			} else {
				// others are just empty or have global type names
				attributeType = arrayedTypeNameForParameter(type.as(CdsArrayedType.class), cfg);
			}
		} else {
			logger.warn("Interface Generation: Unsupported CDS Element with attribute name '{}' and type '{}'.",
					type.getName(), type);
			return null;
		}
		return attributeType;
	}

	public static TypeName getReturnType(CdsElement attribute, Configuration cfg) {
		boolean isMedia = attribute.annotations().anyMatch(a -> "Core.MediaType".equals(a.getName()));
		if (isMedia && attribute.getType().isSimple()) {
			CdsSimpleType simpleType = attribute.getType().as(CdsSimpleType.class);
			return PoetTypeName.getTypeName(CdsBaseType.cdsJavaMediaType(simpleType.getType()).getName());
		}
		if (attribute.getType().isStructured() && attribute.getType().getQualifiedName().isEmpty()) {
			return getTypeName(toUpperCamel(attribute.getName()));
		} else if (isAnonymousAspect(attribute)) {
			TypeName attributeType;
			attributeType = getTypeName(toUpperCamel(attribute.getName()));
			if (!CdsModelUtils.isSingleValued(attribute.getType())) {
				attributeType = listOf(attributeType);
			}
			return attributeType;
		}
		return getAttributeType(attribute.getType(), cfg, null, attribute);
	}

	public static TypeName getOperationResultType(CdsOperation operation, CdsType returnType, NamesUtils namesUtils, Configuration config) {
		if (TypeUtils.isAnonymousType(returnType, config)) {
			ClassName innerEventContextClassName = getInnerEventContextClassName(operation, RETURN_TYPE);
			if (returnType.isArrayed()) {
				return getArrayTypeName(innerEventContextClassName);
			}
			return innerEventContextClassName;
		}

		if (returnType.isStructured()) {
			return getTypeName(namesUtils.qualifiedJavaClassName(returnType));
		} else if (returnType.isSimple()) {
			return getTypeName(returnType.as(CdsSimpleType.class).getJavaType().getTypeName());
		} else if (returnType.isArrayed()) {
			if (config.getSharedInterfaces() && TypeUtils.isAnonymousType(returnType)) {
				return getAttributeType(returnType, config, returnType.getQualifiedName(), null);
			} else {
				TypeName nestedType = getOperationResultType(operation, returnType.as(CdsArrayedType.class).getItemsType(), namesUtils, config);
				return ParameterizedTypeName.get(Types.COLLECTION, nestedType);
			}
		} else {
			logger.warn("Consumption Interface Generation: Unsupported CDS Element with attribute name {} and type {}",
					operation.getName(), returnType.getName());
			return null;
		}
	}

	public static ClassName getInnerEventContextClassName(CdsDefinition def, String innerClassName) {
		return ClassName.bestGuess(String.format("%s.%s", toUpperCamel(def.getName() + "Context"), toUpperCamel(innerClassName)));
	}

	public static ParameterizedTypeName listOf(TypeName type) {
		return ParameterizedTypeName.get(Types.LIST, type);
	}

	public static ClassName className(String name) {
		return ClassName.bestGuess(toUpperCamel(name));
	}

	public static ClassName builderClassName(CdsDefinition def) {
		return ClassName.bestGuess(toUpperCamel(def.getName()) + "_");
	}

	public static boolean isAnonymousAspect(CdsElement element) {
		CdsType type = element.getType();
		if (type.isAssociation()) {
			Optional<CdsStructuredType> targetAspect = type.as(CdsAssociationType.class).getTargetAspect();
			if (targetAspect.isPresent() && targetAspect.get().getQualifiedName().isEmpty()) {
				return true;
			}
		}
		return false;
	}

	public static boolean isAnonymousType(CdsType type, Configuration config) {
		if (type.isArrayed()) {
			boolean isTypeAnonymous = TypeUtils.isAnonymousType(type);
			if (config.getSharedInterfaces()) {
				return isTypeAnonymous && type.getQualifiedName().isEmpty();
			} else {
				return isTypeAnonymous;
			}
		}
		return isAnonymousType(type);
	}

	public static boolean isAnonymousType(CdsType type) {
		if (type == null) {
			return false;
		} else if (type.isArrayed()) {
			return isAnonymousType(type.as(CdsArrayedType.class).getItemsType());
		} else if (type.isStructured() && type.getQualifiedName().isEmpty()) {
			return true;
		}
		return false;
	}

	public static Stream<CdsElement> getAnonymousElements(CdsElement element) {
		CdsType type = element.getType();
		if (type.isAssociation()) {
			Optional<CdsStructuredType> targetAspect = type.as(CdsAssociationType.class).getTargetAspect();
			if (targetAspect.isPresent() && targetAspect.get().getQualifiedName().isEmpty()) {
				return targetAspect.get().as(CdsStructuredType.class).elements();
			}
		}
		return Stream.empty();
	}

	public static void addStaticMethod(TypeSpec.Builder builder, TypeName returnType,
			TypeSpec.Builder... anonymousbuilder) {
		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("create").returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);
		CodeBlock.Builder codeBuilder = CodeBlock.builder();
		codeBuilder.addStatement("return $T.create($T.class)", Types.STRUCT, returnType);

		methodBuilder.addCode(codeBuilder.build());
		if (anonymousbuilder.length > 0) {
			anonymousbuilder[0].addMethod(methodBuilder.build());
		} else {
			builder.addMethod(methodBuilder.build());
		}
	}

	public static void addStaticCreateForKeys(TypeSpec.Builder builder, CdsEntity entity, Map<String, ParameterSpec> parameters) {
		if (!parameters.isEmpty()) {
			ClassName returnType = className(entity.getName());
			MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("create").returns(returnType)
														 .addModifiers(Modifier.PUBLIC, Modifier.STATIC);
			parameters.forEach((name, spec) -> methodBuilder.addParameter(spec));

			CodeBlock.Builder codeBuilder = CodeBlock.builder();
			codeBuilder.addStatement("$T<String, Object> keys = new $T<>()", Types.MAP, Types.HASH_MAP);
			parameters.forEach((name, spec) ->
					codeBuilder.addStatement("keys.put($L, $L)", name, spec.name));
			codeBuilder.addStatement("return $T.access(keys).as($T.class)", Types.STRUCT, returnType);

			methodBuilder.addCode(codeBuilder.build());
			builder.addMethod(methodBuilder.build());
		}
	}


	public static List<CdsElement> getManagedToOneFks(CdsElement attribute) {
		return new OnConditionAnalyzer(attribute, false).getFkMapping().keySet().stream()
				.map(m -> attribute.getDeclaringType().as(CdsStructuredType.class).findElement(m)).map(Optional::get)
				.toList();
	}

	public static void logWarningForManyToManyWithStructElement(CdsModel model, CdsStructuredType struct) {
		if (struct.elements().anyMatch(e -> e.getType().isStructured())) {
			Optional<CdsElement> assocElement = model.entities().flatMap(CdsStructuredType::associations)
					.filter(a -> a.getType().as(CdsAssociationType.class).getTargetAspect()
							.map(ta -> ta.getName().equals(struct.getName())).orElse(false))
					.findFirst();
			if (assocElement.isPresent()) {
				logger.warn(
						"Limited CRUD operation support available with composition of aspects({}) containing structured elements. Use dynamic queries in such cases.",
						struct.getQualifiedName());
			}
		}
	}

	private static TypeName arrayedTypeNameForParameter(CdsArrayedType type, Configuration cfg) {
		CdsType itemsType = type.getItemsType();
		if (itemsType.isSimple()) {
			// Erase the type (global arrayed or global item type if the item type is simple to remain compatible)
			return getArrayTypeName(getTypeName(itemsType.as(CdsSimpleType.class).getJavaType().getTypeName()));
		} else {
			String javaClassName = "";
			if (!type.getQualifiedName().isEmpty() && itemsType.getQualifiedName().isEmpty()) {
				// The arrayed type itself generated as Collection.Item
				return getArrayTypeName(asNameWithItemsPostfix(qualifiedJavaClassName(cfg.getBasePackage(), type.getQualifiedName(), type.getName())));
			} else if (!itemsType.getQualifiedName().isEmpty()) {
				javaClassName = qualifiedJavaClassName(cfg.getBasePackage(), itemsType.getQualifiedName(), itemsType.getName());
			}

			if (javaClassName.isEmpty()) {
				logger.error("Interface Generation: Unsupported combination of empty type and items type for function/action parameter.");
				return null;
			}

			return getArrayTypeName(getTypeName(javaClassName));
		}
	}

	private static TypeName arrayedTypeNameForElement(CdsArrayedType type, Configuration cfg, CdsElement element) {
		CdsType itemsType = type.getItemsType();
		boolean arrayedInlinedStructuredType = type.getQualifiedName().contains(".");
		TypeName innerAttributeType = getAttributeType(itemsType, cfg, !arrayedInlinedStructuredType ? type.getName() : null, element);
		// Arrayed Types which are part of inline defined structured type will not have multiple segment qualifiedName.
		// For such types, we run the else condition.
		if (innerAttributeType.toString().endsWith(".")) {
			if (type.getQualifiedName().startsWith(element.getDeclaringType().getQualifiedName())) {
				String className = toUpperCamel(unqualifiedName(type.getQualifiedName()));
				TypeName outer = getTypeName(
					qualifiedJavaClass(cfg.getBasePackage(), namespace(type.getQualifiedName())));
				return getArrayTypeName(getTypeName(outer + "." + className));
			} else {
				String typeName = qualifiedJavaClassName(cfg.getBasePackage(), type.getQualifiedName(), type.getName());
				return getArrayTypeName(asNameWithItemsPostfix(typeName));
			}
		} else {
			return getArrayTypeName(innerAttributeType);
		}
	}

}
