/*******************************************************************
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.NamesUtils.checkForJavaKeyword;
import static com.sap.cds.generator.util.NamesUtils.getNameIfNotIgnored;
import static com.sap.cds.generator.util.PoetTypeName.getArrayTypeName;
import static com.sap.cds.generator.util.TypeUtils.className;
import static com.sap.cds.generator.util.TypeUtils.getAttributeType;
import static com.sap.cds.generator.util.TypeUtils.getInnerEventContextClassName;
import static com.sap.cds.generator.util.TypeUtils.getOperationResultType;
import static com.sap.cds.generator.util.TypeUtils.isAnonymousType;
import static com.sap.cds.generator.writer.CaseFormatHelper.toLowerCamel;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperUnderscore;
import static com.sap.cds.generator.writer.SpecWriterUtil.cdsNameAnnotation;

import java.util.List;
import java.util.Optional;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.Cds4jCodegen;
import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.generator.util.PoetTypeName;
import com.sap.cds.generator.util.TypeUtils;
import com.sap.cds.impl.util.Pair;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsBoundAction;
import com.sap.cds.reflect.CdsBoundFunction;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsOperation;
import com.sap.cds.reflect.CdsParameter;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class CreateAppServiceInterfaceVisitor implements CdsVisitor {

	private static final String DOT_CDS_NAME = ".CDS_NAME";
	private static final String DRAFT_SUFFIX = "_drafts";
	private static final ClassName INNER_DRAFT = ClassName.get("", "Draft");
	private static final ClassName INNER_APPLICATION = ClassName.get("", "Application");
	private static final ClassName INNER_REMOTE = ClassName.get("", "Remote");
	private final TypeSpec.Builder builder;
	private final Configuration config;
	private final NamesUtils namesUtils;
	private final String interfaceName;

	public static CdsVisitor create(TypeSpec.Builder builder, String interfaceName, Configuration config, NamesUtils namesUtils) {
		return new CreateAppServiceInterfaceVisitor(builder, interfaceName, config, namesUtils);
	}

	CreateAppServiceInterfaceVisitor(TypeSpec.Builder builder, String interfaceName, Configuration config, NamesUtils namesUtils) {
		this.builder = builder;
		this.interfaceName = interfaceName;
		this.config = config;
		this.namesUtils = namesUtils;
	}

	@Override
	public void visit(CdsService service) {
		if (namesUtils.isExcluded(service.getQualifiedName()) || Cds4jCodegen.isIgnored(service)) {
			return;
		}

		checkForJavaKeyword(service.getQualifiedName());

		builder.addSuperinterface(Types.CQN_SERVICE);

		// add annotation @CdsName(<BuilderClassName>.CDS_NAME) at service interface
		String cdsNameValue = TypeUtils.builderClassName(service).simpleName() + DOT_CDS_NAME;
		builder.addAnnotation(cdsNameAnnotation(cdsNameValue, "$L"));

		ClassName name = ClassName.bestGuess(this.interfaceName);
		addInnerInterface(INNER_APPLICATION, new ClassName[] { Types.APPLICATION_SERVICE, name });
		addInnerInterface(INNER_REMOTE, new ClassName[] { Types.REMOTE_SERVICE, name });
		if (isServiceDraftEnabled(service)) {
			addInnerInterface(INNER_DRAFT, new ClassName[] { Types.DRAFT_SERVICE, name });
		}

		service.entities().forEach(entity -> entity.accept(this));
		service.actions().forEach(action -> action.accept(this));
		service.functions().forEach(function -> function.accept(this));
	}

	private void addInnerInterface(ClassName interfaceName, ClassName[] superInterfaces) {
		TypeSpec.Builder innerInterfaceBuilder = TypeSpec.interfaceBuilder(interfaceName).addModifiers(Modifier.PUBLIC,
				Modifier.STATIC);
		List.of(superInterfaces).forEach(innerInterfaceBuilder::addSuperinterface);
		builder.addType(innerInterfaceBuilder.build());
	}

	@Override
	public void visit(CdsEntity entity) {
		if (entity.getName().endsWith(DRAFT_SUFFIX) || namesUtils.isExcluded(entity.getQualifiedName())
				|| Cds4jCodegen.isIgnored(entity)) {
			return;
		}

		checkForJavaKeyword(entity.getQualifiedName());

		entity.actions().forEach(action -> {
			// don't create methods for draft actions
			if (CdsConstants.DRAFT_ACTIONS.contains(action.getName())) {
				return;
			}

			Optional<MethodSpec> method = buildOperationFacadeMethod(action, action.returnType(),
					Optional.of(entity));
			method.ifPresent(builder::addMethod);
		});

		entity.functions().forEach(function -> {
			Optional<MethodSpec> method = buildOperationFacadeMethod(function, function.returnType(),
					Optional.of(entity));
			method.ifPresent(builder::addMethod);
		});
	}

	@Override
	public void visit(CdsFunction function) {
		if (namesUtils.isExcluded(function.getQualifiedName()) || Cds4jCodegen.isIgnored(function)) {
			return;
		}

		Optional<MethodSpec> method = buildOperationFacadeMethod(function, function.returnType(), Optional.empty());
		method.ifPresent(builder::addMethod);
	}

	@Override
	public void visit(CdsAction action) {
		if (namesUtils.isExcluded(action.getQualifiedName()) || Cds4jCodegen.isIgnored(action)) {
			return;
		}

		Optional<MethodSpec> method = buildOperationFacadeMethod(action, action.returnType(), Optional.empty());
		method.ifPresent(builder::addMethod);
	}

	private Optional<MethodSpec> buildOperationFacadeMethod(CdsOperation operation, Optional<CdsType> cdsReturnType,
			Optional<CdsEntity> entityOpt) {
		checkForJavaKeyword(operation.getQualifiedName());

		Optional<String> methodName = NamesUtils.getNameIfNotIgnored(operation, operation.getName());
		if (methodName.isEmpty()) {
			return Optional.empty();
		}

		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(namesUtils.validJavaMethodName(methodName.get()))
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);

		// get effective name of operation, could be renamed with @cds.java.name
		Pair<String, String> effectiveName = NamesUtils.getEffectiveNames(operation);

		// name of corresponding event context class, ends with suffix "Context"
		ClassName eventContextClassName = className(
				effectiveName.left + CreateEventContextInterfaceVisitor.CONTEXT_SUFFIX);
		// add annotation @CdsName(<EventContextClassName>.CDS_NAME)
		methodBuilder.addAnnotation(cdsNameAnnotation(eventContextClassName.simpleName() + DOT_CDS_NAME, "$L"));

		if (cdsReturnType.isPresent()) {
			TypeName returnType = getOperationResultType(operation, cdsReturnType.get(), namesUtils, config);
			methodBuilder.returns(returnType);
		}

		if (entityOpt.isPresent()) {
			CdsEntity entity = entityOpt.get();
			CdsParameter bindingParameter = null;
			if (operation instanceof CdsBoundAction action) {
				bindingParameter = action.getBindingParameter();
			} else if (operation instanceof CdsBoundFunction function) {
				bindingParameter = function.getBindingParameter();
			}

			String bindingParameterName = bindingParameter != null ? bindingParameter.getName() : "ref";

			TypeName builderType = PoetTypeName
					.getTypeName(TypeUtils.builderClassName(entity).simpleName());

			ParameterSpec bindingParam = ParameterSpec.builder(builderType, bindingParameterName).build();
			methodBuilder.addParameter(bindingParam);
		}

		operation.parameters().forEach(param -> {
			Optional<String> paramName = getNameIfNotIgnored(param, param.getName());
			if (paramName.isEmpty()) {
				return;
			}

			TypeName paramType = getParameterType(operation, param);

			// creates a literal pointing to a constant in the event context interface, e.g. AddToOrderContext.ORDER_ID
			String cdsNameValue = eventContextClassName.simpleName() + "." + toUpperUnderscore(param.getName());
			// creates a Java parameter name with lower camel case, e.g. "order_ID" -> "orderId"
			ParameterSpec paramSpec = ParameterSpec.builder(paramType, toLowerCamel(paramName.get()))
					.addAnnotation(cdsNameAnnotation(cdsNameValue, "$L")).build();
			methodBuilder.addParameter(paramSpec);
		});

		return Optional.of(methodBuilder.build());
	}

	private TypeName getParameterType(CdsOperation operation, CdsParameter param) {
		// If the type is anonymous (no global name or the configuration does not let use the global name)
		// the type of the parameter is the name of the interface defined within the operation context
		// Otherwise, the generic type name is used (also could be anonymous or even primitive type name)
		if (isAnonymousType(param.getType(), config)) {
			TypeName innerType = getInnerEventContextClassName(operation, param.getName());
			if (param.getType().isArrayed()) {
				return getArrayTypeName(innerType);
			} else {
				return innerType;
			}
		} else {
			return getAttributeType(param.getType(), config);
		}
	}

	private static boolean isServiceDraftEnabled(CdsService service) {
		return service.entities()
				.anyMatch(entity -> entity.findAnnotation(CdsConstants.ANNOTATION_DRAFT_ENABLED).isPresent());
	}
}
