/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.PoetTypeName.getTypeName;
import static com.sap.cds.generator.util.TypeUtils.builderClassName;
import static com.sap.cds.generator.util.TypeUtils.getAnonymousElements;
import static com.sap.cds.generator.util.TypeUtils.getReturnType;
import static com.sap.cds.generator.util.TypeUtils.isAnonymousAspect;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;
import static com.sap.cds.generator.writer.SpecWriterUtil.addCdsNameAnnotation;
import static com.sap.cds.generator.writer.SpecWriterUtil.addStaticField;
import static javax.lang.model.element.Modifier.ABSTRACT;
import static javax.lang.model.element.Modifier.PUBLIC;

import java.util.Optional;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.generator.util.PoetTypeName;
import com.sap.cds.generator.writer.ModelWriter.Context;
import com.sap.cds.ql.CdsName;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsEvent;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.CdsVisitor;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CreateBuilderInterfaceVisitor implements CdsVisitor {

	private static final Logger logger = LoggerFactory.getLogger(CreateBuilderInterfaceVisitor.class);
	private static final String FILTER = "filter";
	private final TypeSpec.Builder entityClass;
	private final Configuration cfg;
	private final NamesUtils namesUtils;
	private String entityName;
	private final Context context;

	CreateBuilderInterfaceVisitor(TypeSpec.Builder builder, Context context) {
		this.entityClass = builder;
		this.context = context;
		this.namesUtils = context.namesUtils();
		this.cfg = context.config();
	}

	@Override
	public void visit(CdsEntity entity) {
		generateInterface(entity);
	}

	@Override
	public void visit(CdsEvent event) {
		generateInterface(event);
	}

	@Override
	public void visit(CdsStructuredType struct) {
		generateInterface(struct);
	}

	@Override
	public void visit(CdsElement attribute) {
		if (context.isTenantDiscriminator(attribute)) {
			return;
		}

		if (attribute.getType().isStructured() && attribute.getType().as(CdsStructuredType.class).isAnonymous()) {
			addAnonymousStructAttribute(attribute);
		} else if (!attribute.getType().isAssociation()) {
			addAttribute(attribute);
		} else {
			addAssociationAttribute(attribute);
		}
	}

	private void generateInterface(CdsDefinition def) {
		TypeName type = builderClassName(def);

		entityClass.addSuperinterface(ParameterizedTypeName.get(Types.STRUCTURED_TYPE, type));
		entityClass.addAnnotation(
				AnnotationSpec.builder(CdsName.class).addMember("value", "$S", def.getQualifiedName()).build());
		addStaticQualifiedAttribute(def);
		entityName = def.getName();
	}

	private void addAnonymousStructAttribute(CdsElement attribute) {
		String unqualifiedClassName = toUpperCamel(attribute.getName()) + cfg.getClassNameSuffix();

		entityClass.addMethod(MethodSpec.methodBuilder(attribute.getName()).addModifiers(PUBLIC, ABSTRACT)
				.returns(PoetTypeName.getTypeName(unqualifiedClassName)).build());

		//generate inner interface
		Stream<CdsElement> elements = attribute.getType().as(CdsStructuredType.class).elements();
		addInnerInterface(unqualifiedClassName, elements);
	}

	private void addInnerInterface(String unqualifiedClassName, Stream<CdsElement> elements) {
		TypeSpec.Builder innerInterfaceBuilder = TypeSpec.interfaceBuilder(unqualifiedClassName)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);
		innerInterfaceBuilder.addSuperinterface(
				ParameterizedTypeName.get(Types.STRUCTURED_TYPE, PoetTypeName.getTypeName(unqualifiedClassName)));
		elements.forEach(e -> e.accept(new CreateBuilderInterfaceVisitor(innerInterfaceBuilder, context)));
		entityClass.addType(innerInterfaceBuilder.build());
	}

	private void addStaticQualifiedAttribute(CdsDefinition def) {
		String qualifiedName = def.getQualifiedName();

		FieldSpec staticField = FieldSpec.builder(String.class, "CDS_NAME")
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL).initializer("$S", qualifiedName)
				.build();
		entityClass.addField(staticField);
	}

	private void addAssociationAttribute(CdsElement attribute) {
		if (isAnonymousAspect(attribute)) {
			addInnerInterface(toUpperCamel(attribute.getName()) + cfg.getClassNameSuffix(), getAnonymousElements(attribute));
		}
		addMainFunction(attribute);
		addFilterFunction(attribute);
	}

	private void addAttribute(CdsElement attribute) {
		Optional<String> attrName = NamesUtils.getNameIfNotIgnored(attribute, attribute.getName());
		attrName.ifPresent(name -> {
			ParameterizedTypeName returnType = ParameterizedTypeName.get(Types.REFERENCE, getReturnType(attribute, cfg));
			MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(name).addModifiers(PUBLIC, ABSTRACT)
					.returns(returnType);
			if (addCdsNameAnnotation(methodBuilder, attribute, name)) {
				addStaticField(entityClass, attribute);
			}
			entityClass.addMethod(methodBuilder.build());
		});
	}

	private void addMainFunction(CdsElement attribute) {
		String targetEntityClassName = getTargetName(attribute);
		TypeName returnType = PoetTypeName.getTypeName(targetEntityClassName);
		Optional<String> attrName = NamesUtils.getNameIfNotIgnored(attribute, attribute.getName());
		attrName.ifPresentOrElse(
				a -> entityClass.addMethod(
						MethodSpec.methodBuilder(a).addModifiers(PUBLIC, ABSTRACT).returns(returnType).build()),
				() -> logger.error(String.format("Unable to generate method %s for %s", attribute.getName(),
						targetEntityClassName)));
	}

	private void addFilterFunction(CdsElement attribute) {
		if (!attribute.getName().equalsIgnoreCase(FILTER)) {
			String targetEntityClassName = getTargetName(attribute);
			TypeName returnType = PoetTypeName.getTypeName(targetEntityClassName);
			Optional<String> attrName = NamesUtils.getNameIfNotIgnored(attribute, attribute.getName());
			attrName.ifPresentOrElse(
					a -> entityClass.addMethod(
							MethodSpec.methodBuilder(a)
									.addParameter(ParameterizedTypeName.get(Types.FUNCTION, returnType,
											Types.PREDICATE), FILTER).addModifiers(PUBLIC, ABSTRACT).returns(returnType).build()),
					() -> logger.error(String.format("Unable to generate filter method %s for %s", attribute.getName(),
							targetEntityClassName)));
					
		} else {
			logger.warn("There is a name clash between the element 'filter' of entity '{}'"
					+ " and a method in the super interface StructuredType. "
					+ "A filter function accepting predicates will not be generated.", entityName);
		}
	}

	public String getTargetName(CdsElement element) {
		CdsAssociationType association = element.getType();
		CdsType target = association.getTargetAspect().orElseGet(association::getTarget);
		if (target.getQualifiedName().isEmpty()) {
			return toUpperCamel(element.getName()) + cfg.getClassNameSuffix();
		}
		return interfaceName(target);
	}

	private String interfaceName(CdsDefinition def) {
		return getTypeName(namesUtils.qualifiedJavaClassName(def)) + cfg.getClassNameSuffix();
	}
}
