/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.NamesUtils.getNameIfNotIgnored;
import static com.sap.cds.generator.util.NamesUtils.innerInterfaceQualifiedName;
import static com.sap.cds.generator.util.NamesUtils.unqualifiedName;
import static com.sap.cds.generator.util.PoetTypeName.getTypeName;
import static com.sap.cds.generator.util.TypeUtils.addStaticCreateForKeys;
import static com.sap.cds.generator.util.TypeUtils.addStaticMethod;
import static com.sap.cds.generator.util.TypeUtils.builderClassName;
import static com.sap.cds.generator.util.TypeUtils.className;
import static com.sap.cds.generator.util.TypeUtils.getAnonymousElements;
import static com.sap.cds.generator.util.TypeUtils.getManagedToOneFks;
import static com.sap.cds.generator.util.TypeUtils.getReturnType;
import static com.sap.cds.generator.util.TypeUtils.isAnonymousAspect;
import static com.sap.cds.generator.util.TypeUtils.listOf;
import static com.sap.cds.generator.writer.CaseFormatHelper.toLowerCamel;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;
import static com.sap.cds.generator.writer.SpecWriterUtil.addCdsNameAnnotation;
import static com.sap.cds.generator.writer.SpecWriterUtil.addCdsPathAnnotation;
import static com.sap.cds.generator.writer.SpecWriterUtil.addFkStaticField;
import static com.sap.cds.generator.writer.SpecWriterUtil.addStaticField;
import static com.sap.cds.generator.writer.SpecWriterUtil.cdsNameAnnotation;
import static com.sap.cds.generator.writer.SpecWriterUtil.setJavaDoc;
import static com.sap.cds.util.CdsModelUtils.managedToOne;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.MethodStyle;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.generator.writer.ModelWriter.Context;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsEnumType;
import com.sap.cds.reflect.CdsEvent;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.reflect.impl.CdsEventBuilder.EventProxy;
import com.sap.cds.reflect.impl.DraftAdapter;
import com.sap.cds.util.CdsModelUtils;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.TypeSpec.Builder;
import com.squareup.javapoet.WildcardTypeName;

public class CreateConsumptionInterfaceVisitor implements CdsVisitor {
	private static final ParameterizedTypeName MAP_STR2OBJ = ParameterizedTypeName.get(Types.MAP,
			Types.STRING, WildcardTypeName.subtypeOf(Object.class));
	private final TypeSpec.Builder builder;
	private final Configuration cfg;
	private final String typeName;
	private final Context context;

	CreateConsumptionInterfaceVisitor(TypeSpec.Builder builder, String typeName, Context context) {
		this.builder = builder;
		this.cfg = context.config();
		this.typeName = typeName;
		this.context = context;
		if (!builder.superinterfaces.contains(Types.CDS_DATA)) {
			this.builder.addSuperinterface(Types.CDS_DATA);
		}
	}

	@Override
	public void visit(CdsEntity entity) {
		addRefMethod(entity);
		addStaticMethod(builder, className(entity.getName()));

		Map<String, ParameterSpec> keyElements = new LinkedHashMap<>();
		entity.keyElements().forEach(
			key -> getNameIfNotIgnored(key, key.getName()).ifPresent(
					name ->
						keyElements.put(
							CaseFormatHelper.toUpperUnderscore(name),
							ParameterSpec.builder(getSetterParam(key), toLowerCamel(name)).build())));

		// In CSN the keys can be in arbitrary order and the signatures will change when order of elements changes
		// To avoid that, additional create method generated only if entity has single key
		if (keyElements.size() == 1) {
			addStaticCreateForKeys(builder, entity, keyElements);
		}

		builder.addAnnotation(cdsNameAnnotation(entity.getQualifiedName(), "$S"));
	}

	@Override
	public void visit(CdsEvent event) {
		addStaticMethod(builder, getTypeName(toUpperCamel(event.getName())));

		builder.addAnnotation(cdsNameAnnotation(event.getQualifiedName(), "$S"));
	}

	@Override
	public void visit(CdsStructuredType struct) {
		addStaticMethod(builder, getTypeName(toUpperCamel(struct.getName())));

		builder.addAnnotation(cdsNameAnnotation(struct.getQualifiedName(), "$S"));
	}

	@Override
	public void visit(CdsArrayedType type) {
		if (!type.getQualifiedName().isEmpty() && type.getItemsType().isStructured()
			&& (type.getItemsType().getQualifiedName().isEmpty() && !type.getItemsType().isSimple())) {

			builder.superinterfaces.clear(); // Make wrapper interface extend nothing

			TypeSpec.Builder innerIntefaceBuilder = TypeSpec
				.interfaceBuilder(NamesUtils.ITEM_TYPE_NAME).addModifiers(Modifier.PUBLIC, Modifier.STATIC);

			type.getItemsType().as(CdsStructuredType.class).elements().forEach(e ->
				e.accept(new CreateConsumptionInterfaceVisitor(innerIntefaceBuilder, NamesUtils.ITEM_TYPE_NAME, context)));
			addStaticMethod(builder, getTypeName(NamesUtils.ITEM_TYPE_NAME), innerIntefaceBuilder);
			builder.addType(innerIntefaceBuilder.build());
		}
	}

	@Override
	public void visit(CdsElement element) {
		if (context.isTenantDiscriminator(element)) {
			return;
		}
		addStaticField(builder, element);

		if (isAnonymousType(element)) {
			String unqualifiedClassName = unqualifiedName(toUpperCamel(element.getName()));
			TypeSpec.Builder innerIntefaceBuilder = TypeSpec.interfaceBuilder(unqualifiedClassName)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);

			CdsType type = element.getType();
			Stream<CdsElement> elements;
			if (type.isStructured() && type.getQualifiedName().isEmpty()) {
				elements = type.as(CdsStructuredType.class).elements();
			} else if (type.isArrayed()) {
				elements = type.as(CdsArrayedType.class).getItemsType().as(CdsStructuredType.class).elements();
			} else {
				elements = getAnonymousElements(element);
			}
			String qualifiedElementName = innerInterfaceQualifiedName(element, cfg.getBasePackage());
			elements.forEach(e -> e
				.accept(new CreateConsumptionInterfaceVisitor(innerIntefaceBuilder, qualifiedElementName, context)));
			addStaticMethod(builder, getTypeName(toUpperCamel(unqualifiedClassName)), innerIntefaceBuilder);
			populateGetter(element, getReturnType(element, cfg), false);
			populateSetter(element, getReturnType(element, cfg), false);
			builder.addType(innerIntefaceBuilder.build());
		} else if (element.getType().isEnum()) {
			generateInnerConstantClass(builder, element);
			getter(element);
			setter(element);
		} else {
			getter(element);
			setter(element);
		}
		if (cfg.fkAccessors() && managedToOne(element.getType()) && !element.getName().equals(DraftAdapter.DRAFT_ADMINISTRATIVE_DATA)) {
			addFkStaticField(builder, element);
			addFkMethods(element);
		}

	}

	private void generateInnerConstantClass(Builder builder, CdsElement element) {
		CdsEnumType<?> type = element.getType().as(CdsEnumType.class);
		// Anonymous enum type name is equal to the type of the items of it
		if (type.getQualifiedName().equals(type.getType().cdsName())) {
			String innerTypeName = element.getName();
			TypeSpec.Builder result = TypeSpec.classBuilder(className(innerTypeName));
			type.accept(CreateEnumConstantClassVisitor.create(result));
			if (!result.fieldSpecs.isEmpty()) {
				// JavaPoet always wants STATIC modifier for inner types
				result.addModifiers(Modifier.STATIC);
				builder.addType(result.build());
			}
		}
	}

	private void addRefMethod(CdsEntity entity) {
		if (!entity.isAbstract()) {
			TypeName returnType = builderClassName(entity);
			MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("ref").returns(returnType)
					.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);

			builder.addMethod(methodBuilder.build());
		}
	}

	private boolean isAnonymousType(CdsElement element) {
		CdsType type = element.getType();
		if (type.isArrayed()) {
			if (type.getQualifiedName().startsWith(element.getDeclaringType().getQualifiedName())) {
				CdsType itemsType = type.as(CdsArrayedType.class).getItemsType();
				return itemsType.isStructured() && itemsType.getQualifiedName().isEmpty();
			} else {
				return false;
			}
		} else if (type.isStructured() && type.getQualifiedName().isEmpty()) {
			return true;
		} else return isAnonymousAspect(element);
	}

	private void setter(CdsElement attribute) {
		TypeName paramType = getSetterParam(attribute);

		if (paramType == null) {
			return;
		}
		populateSetter(attribute, paramType, false);
	}

	private void populateSetter(CdsElement attribute, TypeName paramType, boolean cdsPath) {
		Optional<String> attributeName = getNameIfNotIgnored(attribute, attribute.getName());
		attributeName.ifPresent(name -> {
			String setter;
		    TypeName returnType;

			if (cfg.getMethodStyle() == MethodStyle.BEAN) {
				returnType = TypeName.VOID;
				setter = "set" + toUpperCamel(name);
			} else {
				returnType = getTypeName(typeName);
				setter = toLowerCamel(name);
			}

			MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(setter).returns(returnType)
					.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT).addParameter(paramType, toLowerCamel(name));
			addJavaDoc(attribute, methodBuilder);
			if (cdsPath) {
				addCdsPathAnnotation(methodBuilder, attribute.getName(), setter);
			} else {
				addCdsNameAnnotation(methodBuilder, attribute, setter);
			}
			builder.addMethod(methodBuilder.build());
		});
	}

	private TypeName getSetterParam(CdsElement element) {
		if (element.getType().isAssociation()) {
			if (CdsModelUtils.isSingleValued(element.getType())) {
				return MAP_STR2OBJ;
			}
			return listOf(WildcardTypeName.subtypeOf(MAP_STR2OBJ));
		}
		return getReturnType(element, cfg);
	}

	private void getter(CdsElement attribute) {
		TypeName returnType = getReturnType(attribute, cfg);

		if (returnType == null) {
			return;
		}
		populateGetter(attribute, returnType, false);
	}

	private void populateGetter(CdsElement attribute, TypeName returnType, boolean cdsPath) {
		Optional<String> attributeName = getNameIfNotIgnored(attribute, attribute.getName());
		attributeName.ifPresent(name -> {
			String getter;
			if (cfg.getMethodStyle().equals(MethodStyle.BEAN)) {
				getter = "get" + toUpperCamel(name);
			} else {
				getter = toLowerCamel(name);
			}
			MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(getter).returns(returnType)
					.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);
			addJavaDoc(attribute, methodBuilder);
			if (cdsPath) {
				addCdsPathAnnotation(methodBuilder, attribute.getName(), getter);
			} else {
				addCdsNameAnnotation(methodBuilder, attribute, getter);
			}
			builder.addMethod(methodBuilder.build());
		});
	}

	private void addJavaDoc(CdsElement attribute, MethodSpec.Builder methodBuilder) {
		if (!(attribute.getDeclaringType() instanceof EventProxy) && cfg.getDocs()) {
			setJavaDoc(attribute, methodBuilder);
		}
	}

	private void addFkMethods(CdsElement attribute) {
		// Add setter for managed to-one FK
		getManagedToOneFks(attribute).forEach(fkElement -> {
			TypeName keyReturnType = getReturnType(fkElement, cfg);
			if (keyReturnType == null) {
				return;
			}
			populateSetter(fkElement, keyReturnType, true);
			populateGetter(fkElement, keyReturnType, true);
		});
	}
}
