/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.NamesUtils.qualifiedJavaClass;
import static com.sap.cds.generator.util.NamesUtils.unqualifiedName;
import static com.sap.cds.generator.util.PoetTypeName.getArrayTypeName;
import static com.sap.cds.generator.util.PoetTypeName.getTypeName;
import static com.sap.cds.generator.util.TypeUtils.addStaticMethod;
import static com.sap.cds.generator.util.TypeUtils.getAttributeType;
import static com.sap.cds.generator.writer.CaseFormatHelper.lowercaseFirst;
import static com.sap.cds.generator.writer.CaseFormatHelper.toLowerCamel;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperUnderscore;
import static com.sap.cds.generator.writer.SpecWriterUtil.setJavaDoc;
import static com.sap.cds.generator.writer.Types.RETURN_TYPE;
import static com.sap.cds.reflect.impl.CdsAnnotatableImpl.removeAt;
import static com.sap.cds.reflect.impl.reader.model.CdsConstants.ANNOTATION_CDS_JAVA_NAME;

import java.util.Optional;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.MethodStyle;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.generator.util.TypeUtils;
import com.sap.cds.generator.writer.ModelWriter.Context;
import com.sap.cds.ql.CdsName;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEvent;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsOperation;
import com.sap.cds.reflect.CdsParameter;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.util.CdsModelUtils;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.WildcardTypeName;

public class CreateEventContextInterfaceVisitor implements CdsVisitor {

	private static final ParameterizedTypeName MAP_STR2OBJ = ParameterizedTypeName.get(Types.MAP,
			Types.STRING, WildcardTypeName.subtypeOf(Object.class));
	static final String CONTEXT_SUFFIX = "Context";
	private static final String PROXYMETHOD_ARG_NAME = "entityName";
	private final TypeSpec.Builder builder;
	private final Configuration cfg;
	private final boolean bounded;
	private final NamesUtils namesUtils;
	private final String boundEntityName;
	private final String definitionName;
	private final Context context;

	CreateEventContextInterfaceVisitor(TypeSpec.Builder builder, String boundEntityName, String definitionName,
			Context context) {
		this.builder = builder;
		this.cfg = context.config();
		this.boundEntityName = boundEntityName;
		this.definitionName = qualifiedJavaClass(cfg.getBasePackage(), definitionName);
		this.builder.addSuperinterface(Types.EVENT_CONTEXT);
		this.bounded = boundEntityName != null;
		this.namesUtils = context.namesUtils();
		this.context = context;
	}

	@Override
	public void visit(CdsAction action) {
		addGetCqnMethod();
		addSetCqnMethod();
		addStaticQualifiedAttribute(action);
		prepareDefinition(action);
	}

	@Override
	public void visit(CdsFunction function) {
		addGetCqnMethod();
		addSetCqnMethod();
		addStaticQualifiedAttribute(function);
		prepareDefinition(function);
	}

	@Override
	public void visit(CdsEvent event) {
		addGetter(event);
		addSetter(event);
		addStaticQualifiedAttribute(event);
		addStaticProxyMethod(event.getName());
		builder.addAnnotation(eventNameAnnotation(event.getName(), "$S"));
	}

	private void addGetter(CdsEvent event) {
		String methodName;

		if (cfg.getMethodStyle() == MethodStyle.BEAN) {
			methodName = "getData";
		} else {
			methodName = "data";
		}
		MethodSpec.Builder resultGetterBuilder = MethodSpec.methodBuilder(methodName)
				.returns(getTypeName(toUpperCamel(event.getName()))).addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);
		builder.addMethod(resultGetterBuilder.build());
	}

	private void addSetter(CdsEvent event) {
		String methodName;
		TypeName returnType = TypeName.VOID;

		if (cfg.getMethodStyle() == MethodStyle.BEAN) {
			methodName = "setData";
		} else {
			methodName = "data";
			returnType = getTypeName(toUpperCamel(event.getName() + CONTEXT_SUFFIX));
		}
		MethodSpec.Builder resultGetterBuilder = MethodSpec.methodBuilder(methodName).returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT)
				.addParameter(getTypeName(toUpperCamel(event.getName())), "event");
		builder.addMethod(resultGetterBuilder.build());
	}

	private void addGetCqnMethod() {
		if (bounded) {
			String methodName;

			if (cfg.getMethodStyle() == MethodStyle.BEAN) {
				methodName = "getCqn";
			} else {
				methodName = "cqn";
			}
			MethodSpec.Builder resultGetterBuilder = MethodSpec.methodBuilder(methodName)
					.returns(Types.CQN_SELECT)
					.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);
			builder.addMethod(resultGetterBuilder.build());
		}
	}

	private void addSetCqnMethod() {
		if (bounded) {
			String methodName = "cqn";
			TypeName returnType;

			if (cfg.getMethodStyle() == MethodStyle.BEAN) {
				returnType = TypeName.VOID;
				methodName = "setCqn";
			} else {
				returnType = getTypeName(toUpperCamel(unqualifiedName(definitionName) + CONTEXT_SUFFIX));
			}
			MethodSpec.Builder resultGetterBuilder = MethodSpec.methodBuilder(methodName).returns(returnType)
					.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT)
					.addParameter(Types.CQN_SELECT, "select");
			builder.addMethod(resultGetterBuilder.build());
		}
	}

	private void addStaticQualifiedAttribute(CdsDefinition def) {
		String name = def.getName();

		FieldSpec staticField = FieldSpec.builder(String.class, "CDS_NAME")
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL).initializer("$S", name).build();
		builder.addField(staticField);
	}

	private void prepareDefinition(CdsOperation operation) {
		String defName = operation.getAnnotationValue(removeAt(ANNOTATION_CDS_JAVA_NAME), operation.getName());
		if (bounded) {
			addStaticProxyMethodBound(defName);
		}
		addResultMethod(operation);
		addStaticProxyMethod(defName);
		builder.addAnnotation(eventNameAnnotation(operation.getName(), "$S"));
	}

	private void addResultMethod(CdsOperation operation) {
		TypeName resultType = null;
		TypeName setReturnType;

		Optional<CdsType> opt = operation.returnType();
		if (opt.isEmpty()) {
			return;
		}
		CdsType returnType = opt.get();

		if (TypeUtils.isAnonymousType(returnType, cfg)) {
			resultType = generateInnerInterface(returnType, RETURN_TYPE);
		} else {
			resultType = getAttributeType(returnType, cfg);
		}
		if (resultType == null) {
			resultType = TypeUtils.getOperationResultType(operation, returnType, namesUtils, cfg);
			if (resultType == null) {
				// Since actions might or might not return a value
				return;
			}
		}
		String setter;
		String getter;
		if (cfg.getMethodStyle() == MethodStyle.BEAN) {
			setReturnType = TypeName.VOID;
			setter = "setResult";
			getter = "getResult";
		} else {
			setReturnType = getTypeName(toUpperCamel(operation.getName() + CONTEXT_SUFFIX));
			setter = getter = "result";
		}

		MethodSpec.Builder resultSetterBuilder = MethodSpec.methodBuilder(setter).returns(setReturnType)
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT).addParameter(resultType, toLowerCamel("result"));

		MethodSpec.Builder resultGetterBuilder = MethodSpec.methodBuilder(getter).returns(resultType)
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);
		builder.addMethod(resultSetterBuilder.build());
		builder.addMethod(resultGetterBuilder.build());
	}

	private TypeName generateInnerInterface(CdsType returnType, String returnTypeName) {
		Stream<CdsElement> elements;
		TypeName resultType = getTypeName(returnTypeName);

		TypeSpec.Builder innerIntefaceBuilder = TypeSpec.interfaceBuilder(returnTypeName).addModifiers(Modifier.PUBLIC,
				Modifier.STATIC);
		if (returnType.isStructured()) {
			elements = returnType.as(CdsStructuredType.class).elements();
		} else {
			resultType = getArrayTypeName(resultType);
			elements = returnType.as(CdsArrayedType.class).getItemsType().as(CdsStructuredType.class).elements();
		}
		elements.forEach(
				e -> e.accept(new CreateConsumptionInterfaceVisitor(innerIntefaceBuilder, returnTypeName, context)));
		addStaticMethod(builder, getTypeName(toUpperCamel(returnTypeName)), innerIntefaceBuilder);
		builder.addType(innerIntefaceBuilder.build());
		return resultType;
	}

	private void addStaticProxyMethod(String entityName) {
		TypeName returnType = getTypeName(toUpperCamel(entityName + CONTEXT_SUFFIX));
		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("create").returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);
		if (bounded) {
			methodBuilder.addParameter(TypeName.get(String.class), PROXYMETHOD_ARG_NAME);
		}

		CodeBlock.Builder codeBuilder = CodeBlock.builder();
		codeBuilder.addStatement("return EventContext.create($T.class, $N)", returnType,
				bounded ? PROXYMETHOD_ARG_NAME : "null");

		methodBuilder.addCode(codeBuilder.build());
		builder.addMethod(methodBuilder.build());
	}

	private void addStaticProxyMethodBound(String entityName) {
		TypeName returnType = getTypeName(toUpperCamel(entityName + CONTEXT_SUFFIX));

		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("create").returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);
		CodeBlock.Builder codeBuilder = CodeBlock.builder();
		codeBuilder.addStatement("return EventContext.create($T.class, $S)", returnType, this.boundEntityName);

		methodBuilder.addCode(codeBuilder.build());
		builder.addMethod(methodBuilder.build());
	}

	private static AnnotationSpec eventNameAnnotation(String cdsName, String format) {
		return AnnotationSpec.builder(Types.EVENT_NAME).addMember("value", format, cdsName).build();
	}

	@Override
	public void visit(CdsParameter parameter) {
		addStaticAttribute(parameter);
		TypeName resultType;
		CdsType returnType = parameter.getType();
		if(TypeUtils.isAnonymousType(returnType, cfg)) {
			resultType = generateInnerInterface(returnType, unqualifiedName(toUpperCamel(parameter.getName())));
			addGetterMethod(parameter, resultType);
			addSetterMethod(parameter, resultType);
		} else {
			addParamGetter(parameter);
			addParamSetter(parameter);
		}
	}

	private void addParamGetter(CdsParameter parameter) {
		TypeName returnType = getAttributeType(parameter.getType(), cfg);
		addGetterMethod(parameter, returnType);
	}

	private void addGetterMethod(CdsParameter parameter, TypeName returnType) {
		if (returnType == null) {
			return;
		}
		Optional<String> parameterName = NamesUtils.getNameIfNotIgnored(parameter, parameter.getName());
		parameterName.ifPresent(name -> {
			String getter;
			if (cfg.getMethodStyle().equals(MethodStyle.BEAN)) {
				getter = "get" + toUpperCamel(name);
			} else {
				getter = toLowerCamel(name);
			}
			MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(getter).returns(returnType)
					.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);
			addJavaDoc(parameter, methodBuilder);
			addCdsNameAnnotation(methodBuilder, parameter.getName(), getter);
			builder.addMethod(methodBuilder.build());
		});

	}

	private void addParamSetter(CdsParameter parameter) {
		TypeName paramType = getSetterParam(parameter.getType());
		addSetterMethod(parameter, paramType);
	}

	private void addSetterMethod(CdsParameter parameter, TypeName paramType) {
		if (paramType == null) {
			return;
		}

		Optional<String> parameterName = NamesUtils.getNameIfNotIgnored(parameter, parameter.getName());
		parameterName.ifPresent(name -> {
			String setter;
			TypeName returnType;
			if (cfg.getMethodStyle() == MethodStyle.BEAN) {
				returnType = TypeName.VOID;
				setter = "set" + toUpperCamel(name);
			} else {
				returnType = getTypeName(toUpperCamel(unqualifiedName(definitionName) + CONTEXT_SUFFIX));
				setter = toLowerCamel(name);
			}

			MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(setter).returns(returnType)
					.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT)
					.addParameter(paramType, toLowerCamel(name));
			addJavaDoc(parameter, methodBuilder);
			addCdsNameAnnotation(methodBuilder, parameter.getName(), setter);
			builder.addMethod(methodBuilder.build());
		});
	}

	private TypeName getSetterParam(CdsType type) {
		if (type.isAssociation()) {
			if (CdsModelUtils.isSingleValued(type)) {
				return MAP_STR2OBJ;
			}
			return TypeUtils.listOf(WildcardTypeName.subtypeOf(MAP_STR2OBJ));
		}
		return getAttributeType(type, cfg);
	}

	private static AnnotationSpec cdsNameAnnotation(String cdsName, String format) {
		return AnnotationSpec.builder(CdsName.class).addMember("value", format, cdsName).build();
	}

	private static MethodSpec.Builder addCdsNameAnnotation(MethodSpec.Builder builder, String cdsName,
			String javaName) {
		if (!cdsName.equals(propertyName(javaName))) {
			builder.addAnnotation(cdsNameAnnotation(toUpperUnderscore(cdsName), "$L"));
		}
		return builder;
	}

	private static String propertyName(String javaName) {
		if (javaName.startsWith("set") || javaName.startsWith("get")) {
			javaName = lowercaseFirst(javaName.substring(3));
		}
		return javaName;
	}

	private void addStaticAttribute(CdsParameter parameter) {
		String name = toUpperUnderscore(parameter.getName());
		FieldSpec staticField = FieldSpec.builder(String.class, name)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL).initializer("$S", parameter.getName())
				.build();
		builder.addField(staticField);
	}

	private void addJavaDoc(CdsParameter param, MethodSpec.Builder methodBuilder) {
		if (cfg.getDocs()) {
			setJavaDoc(param, methodBuilder);
		}
	}
}
