/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.writer;

import static com.google.common.collect.Streams.concat;
import static com.sap.cds.generator.util.NamesUtils.checkForJavaKeyword;
import static com.sap.cds.generator.util.NamesUtils.getNameIfNotIgnored;
import static com.sap.cds.generator.util.NamesUtils.getResolvedWrapperName;
import static com.sap.cds.generator.util.NamesUtils.isValidTechnicalEntity;
import static com.sap.cds.generator.util.NamesUtils.qualifiedContextname;
import static com.sap.cds.generator.util.NamesUtils.qualifiedWrapperBuilderName;
import static com.sap.cds.generator.util.NamesUtils.unqualifiedName;
import static com.sap.cds.generator.util.TypeUtils.builderClassName;
import static com.sap.cds.generator.util.TypeUtils.className;
import static com.sap.cds.generator.util.TypeUtils.logWarningForManyToManyWithStructElement;
import static com.sap.cds.generator.writer.SpecWriterUtil.cdsNameAnnotation;
import static com.sap.cds.generator.writer.SpecWriterUtil.getJavaDoc;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import javax.lang.model.element.Modifier;
import javax.tools.JavaFileObject;

import com.sap.cds.generator.Cds4jCodegen;
import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.util.EntityWriterException;
import com.sap.cds.generator.util.GeneratedAnnotationUtil;
import com.sap.cds.generator.util.GeneratedFile;
import com.sap.cds.generator.util.GeneratedFile.Consumer;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.generator.util.TypeUtils;
import com.sap.cds.generator.util.UnSupportedModelException;
import com.sap.cds.impl.util.Pair;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsEnumType;
import com.sap.cds.reflect.CdsEvent;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsKind;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.reflect.impl.CdsEventBuilder.EventProxy;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.TypeSpec.Builder;

public class ModelWriter implements CdsVisitor {

	private static final String CONTEXT_SUFFIX = "Context";
	private final Consumer consumer;
	private final Configuration config;
	private final CdsModel model;
	private final Map<String, CdsDefinition> wrapperMap;
	private final Set<String> builderInterfaces;
	private final NamesUtils namesUtils;
	private final GeneratedAnnotationUtil generated;
	private final List<String> generatedServices = new ArrayList<>();
	private final Context context;

	public ModelWriter(Consumer consumer, Configuration config, CdsModel model) {
		this.consumer = consumer;
		this.config = config;
		this.model = model;
		this.wrapperMap = new HashMap<>();
		this.builderInterfaces = new HashSet<>();
		this.namesUtils = new NamesUtils(config);
		this.generated = new GeneratedAnnotationUtil(config);
		this.context = new Context(model.getMeta(CdsConstants.META_TENANT_DISCRIMINATOR), this.config, this.namesUtils);
	}

	@Override
	public void visit(CdsModel model) {
		wrapperMap.values().forEach(this::generateWrapperInterface);
	}

	@Override
	public void visit(CdsEntity entity) {
		checkForJavaKeyword(entity.getQualifiedName());
		if (!namesUtils.isExcluded(entity.getQualifiedName())
				&& isValidTechnicalEntity(model, entity.getQualifiedName())) {
			collectWrapperInterfaces(entity);
			generateBuilderInterface(entity);
			generateConsumptionInterface(entity);
			if (config.getEventContext()) {
				concat(entity.actions(), entity.functions()).forEach(a -> generateEventContextInterface(a, entity));
			}
		}
	}

	@Override
	public void visit(CdsEvent event) {
		checkForJavaKeyword(event.getQualifiedName());
		if (!namesUtils.isExcluded(event.getQualifiedName())) {
			collectWrapperInterfaces(event);
			generateBuilderInterface(event);
			generateConsumptionInterface(event);
			if (config.getEventContext()) {
				generateEventContextInterface(event, null);
			}
		}
	}

	@Override
	public void visit(CdsAction action) {
		if (config.getEventContext() && !namesUtils.isExcluded(action.getQualifiedName())) {
			collectWrapperInterfaces(action);
			generateEventContextInterface(action, null);
		}
	}

	@Override
	public void visit(CdsFunction function) {
		if (config.getEventContext() && !namesUtils.isExcluded(function.getQualifiedName())) {
			collectWrapperInterfaces(function);
			generateEventContextInterface(function, null);
		}
	}

	@Override
	public void visit(CdsArrayedType type) {
		// Explicit types for Arrayed types are required only if the inner type is
		// anonymous and not simple.
		// Explicit inner type e.g. many Books will be represented as Collection<Books>
		// Simple inner type will be represented as Collection<Integer>
		if (!type.getQualifiedName().isEmpty() && type.getItemsType().isStructured()
				&& (type.getItemsType().getQualifiedName().isEmpty() && !type.getItemsType().isSimple())) {

			checkForJavaKeyword(type.getQualifiedName());
			if (!namesUtils.isExcluded(type.getQualifiedName())) {
				generateConsumptionInterface(type);
			}
		}
	}

	@Override
	public void visit(CdsStructuredType struct) {
		if (!struct.getName().isEmpty()) {
			checkForJavaKeyword(struct.getQualifiedName());
			if (!namesUtils.isExcluded(struct.getQualifiedName())) {
				generateConsumptionInterface(struct);
				if (isTargetAspect(struct)) {
					// For every TargetAspect which contains structured elements and is target of
					// many to many composition, throw warning
					logWarningForManyToManyWithStructElement(model, struct);
					generateBuilderInterface(struct);
				}
			}
		}
	}

	@Override
	public void visit(CdsService service) {
		generateTypedAppServiceInterface(service);
	}

	@Override
	public void visit(CdsEnumType<?> type) {
		if (!namesUtils.isExcluded(type.getQualifiedName())) {
			getNameIfNotIgnored(type, type.getName())
					.ifPresent(name -> {
						TypeSpec.Builder result = TypeSpec.classBuilder(TypeUtils.className(name))
								.addModifiers(Modifier.PUBLIC, Modifier.FINAL);
						result.addAnnotation(cdsNameAnnotation(type.getQualifiedName(), "$S"));
						type.accept(CreateEnumConstantClassVisitor.create(result));
						if (!result.fieldSpecs.isEmpty()) {
							generated.addTo(result);
							writeType(namesUtils.packageName(type), result.build());
						}
					});
		}
	}

	public List<String> getGeneratedServices() {
		return this.generatedServices;
	}

	private void collectWrapperInterfaces(CdsDefinition def) {
		String packageName = namesUtils.packageName(def);
		wrapperMap.put(packageName, def);
	}

	private void generateWrapperInterface(CdsDefinition def) {
		String qualifiedName = def.getQualifiedName();
		String name = def.getName();
		String packageName = namesUtils.packageName(def);
		String qualifiedWrapperName = qualifiedWrapperBuilderName(def, config.getClassNameSuffix(), true);
		while (builderInterfaces.contains(qualifiedWrapperName.toLowerCase(Locale.US))) {
			qualifiedWrapperName = getResolvedWrapperName(qualifiedWrapperName, config.getClassNameSuffix());
		}

		Builder builder = TypeSpec.interfaceBuilder(unqualifiedName(qualifiedWrapperName))
				.addModifiers(Modifier.PUBLIC);

		generated.addTo(builder);

		if (WrapperInterfaceCreator.create(builder, model, config)
				.generateInterface(qualifiedContextname(qualifiedName, name))) {
			writeType(packageName, builder.build());
		}
	}

	private void generateConsumptionInterface(CdsDefinition def) {
		Builder builder = createInterfaceSpecBuilder(className(def.getName()));
		addJavadoc(builder, def, () -> !(def instanceof EventProxy) && config.getDocs());

		String qualifiedTypeName = namesUtils.qualifiedJavaClassName(def);
		def.accept(new CreateConsumptionInterfaceVisitor(builder, qualifiedTypeName, context));

		def.getAnnotationValue(CdsConstants.ANNOTATION_JAVA_EXTENDS.substring(1), Collections.<String>emptyList())
				.stream().map(model::getStructuredType).map(this::throwErrorOnExtendEntity)
				.map(namesUtils::qualifiedJavaClassName).map(ClassName::bestGuess).forEach(builder::addSuperinterface);

		generated.addTo(builder);

		TypeSpec typeSpec = builder.build();
		String packageName = namesUtils.packageName(def);
		writeType(packageName, typeSpec);
	}

	private boolean isFuncOrAction(CdsDefinition def) {
		return def.getKind().equals(CdsKind.ACTION) || def.getKind().equals(CdsKind.FUNCTION);
	}

	private void generateEventContextInterface(CdsDefinition def, CdsDefinition boundEntity) {
		if (Cds4jCodegen.isIgnored(def)) {
			return;
		}

		String defName = def.getName();
		String defQualifiedName = def.getQualifiedName();

		if (isFuncOrAction(def)) {
			Pair<String, String> qualifiedName = NamesUtils.getEffectiveNames(def);
			defName = qualifiedName.left;
			defQualifiedName = qualifiedName.right;
		}

		Builder builder = createInterfaceSpecBuilder(className(defName + CONTEXT_SUFFIX));
		addJavadoc(builder, def, config::getDocs);

		String boundEntityName = boundEntity == null ? null : boundEntity.getQualifiedName();

		def.accept(new CreateEventContextInterfaceVisitor(builder, boundEntityName, defQualifiedName, context));

		generated.addTo(builder);

		TypeSpec typeSpec = builder.build();

		String packageName = boundEntity != null ? namesUtils.packageName(boundEntity) : namesUtils.packageName(def);
		writeType(packageName, typeSpec);
	}

	private CdsStructuredType throwErrorOnExtendEntity(CdsStructuredType struct) {
		if (struct.getKind().equals(CdsKind.ENTITY) && !struct.as(CdsEntity.class).isAbstract()) {
			throw new UnSupportedModelException(
					"The '@cds.java.extends' annotation does not support extending an entity.");
		}
		return struct;
	}

	private void generateBuilderInterface(CdsDefinition def) {
		Builder builder = createInterfaceSpecBuilder(builderClassName(def));
		addJavadoc(builder, def, config::getDocs);

		def.accept(new CreateBuilderInterfaceVisitor(builder, context));

		builderInterfaces
				.add(qualifiedWrapperBuilderName(def, config.getClassNameSuffix(), false).toLowerCase(Locale.US));

		generated.addTo(builder);

		TypeSpec typeSpec = builder.build();
		String packageName = namesUtils.packageName(def);
		writeType(packageName, typeSpec);
	}

	private void generateTypedAppServiceInterface(CdsService def) {
		// Classes generated by event context generation are required for action method
		// parameters and result type
		if (!config.getEventContext() || !config.getCqnServices() || namesUtils.isExcluded(def.getQualifiedName())
				|| Cds4jCodegen.isIgnored(def)) {
			return;
		}

		String qualifiedAppServiceInterfaceName = namesUtils.getQualifiedAppServiceInterfaceName(def);
		String interfaceName = unqualifiedName(qualifiedAppServiceInterfaceName);
		Builder builder = TypeSpec.interfaceBuilder(interfaceName).addModifiers(Modifier.PUBLIC);

		addJavadoc(builder, def, config::getDocs);
		generated.addTo(builder);

		def.accept(CreateAppServiceInterfaceVisitor.create(builder, interfaceName, config, namesUtils));

		TypeSpec typeSpec = builder.build();
		String packageName = NamesUtils.namespace(qualifiedAppServiceInterfaceName);
		writeType(packageName, typeSpec);
		this.generatedServices.add(qualifiedAppServiceInterfaceName);
	}

	private void addJavadoc(Builder builder, CdsDefinition def, Supplier<Boolean> doRead) {
		if (doRead.get()) {
			getJavaDoc(def).ifPresent(a -> builder.addJavadoc(a.replace("$", "$$")));
		}
	}

	private static Builder createInterfaceSpecBuilder(ClassName type) {
		return TypeSpec.interfaceBuilder(type).addModifiers(Modifier.PUBLIC);
	}

	private boolean isTargetAspect(CdsStructuredType struct) {
		return model.entities().flatMap(CdsStructuredType::associations)
				.map(a -> a.getType().as(CdsAssociationType.class).getTargetAspect())
				.anyMatch(ta -> ta.isPresent() && ta.get().getName().equals(struct.getName()));
	}

	private void writeType(String packageName, TypeSpec typeSpec) {
		final JavaFile javaFile = JavaFile.builder(packageName, typeSpec).build();
		final JavaFileObject fileObject = javaFile.toJavaFileObject();
		GeneratedFile jpaFile = new GeneratedFile() {
			@Override
			public URI getUri() {
				return fileObject.toUri();
			}

			@Override
			public InputStream getContent() throws IOException {
				return fileObject.openInputStream();
			}
		};
		try {
			consumer.accept(jpaFile);
		} catch (IOException e) {
			String message = String.format("Exception while writing to file %s.", jpaFile.getUri());
			throw new EntityWriterException(message, e);
		}
	}

	/**
	 * A context to hold the tenant discriminator, configuration and utilities.
	 */
	record Context(String tenantDiscriminator, Configuration config, NamesUtils namesUtils) {

		/**
		 * Check if the given element is the tenant discriminator.
		 * 
		 * @param element the {@link CdsElement} to check
		 * @return {@code true} if the given element is a tenant discriminator, {@code false} otherwise
		 */
		boolean isTenantDiscriminator(CdsElement element) {
			if (tenantDiscriminator == null || element == null) {
				return false;
			}
			return tenantDiscriminator.equals(element.getName());
		}
	}
}
