/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator;

import static java.nio.charset.StandardCharsets.UTF_8;

import java.io.IOException;
import java.nio.file.NoSuchFileException;

import com.sap.cds.generator.util.GeneratorMode;
import com.sap.cds.generator.util.ParserMode;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.impl.CdsModelReader;
import com.sap.cds.reflect.impl.CdsModelReader.Config;
import com.sap.cds.reflect.impl.reader.issuecollector.IssueCollector;
import com.sap.cds.reflect.impl.reader.issuecollector.IssueCollectorFactory;
import com.sap.cds.reflect.impl.reader.issuecollector.IssueType;

import org.slf4j.Logger;

/**
 * Helper class to run the code generation.
 */
class Cds4jCodegenRunner {

	private final Logger logger;
	private final IssueCollector issueCollector;

	Cds4jCodegenRunner(Logger logger, IssueCollector issueCollector) {
		this.logger = logger;
		this.issueCollector = issueCollector;
	}

	Result generate(CsnSupplier csn, Generator generator) {
		IssueCollectorFactory.clearIssues();
		IssueCollectorFactory.enableUnrecognizedLogging();
		Result result = new Result();
		try {
			Config config = new CdsModelReader.Config.Builder().setReadDocs(true).build();
			CdsModel cdsModel = CdsModelReader.read(config, new String(csn.get(), UTF_8), false);
			result.status = generator.apply(cdsModel);
		} catch (NoSuchFileException e) {
			String msg = "CSN file not found: ";
			logger.debug(msg, e);
			issueCollector.critical("", msg + e.getFile());
		} catch (Throwable t) { // NOSONAR
			logger.error("Stopped execution due to exception: ", t);
			issueCollector.critical("", t.getMessage());
		}
		result.issues.addAll(IssueCollectorFactory.getIssues());
		return result;
	}

	@FunctionalInterface
	interface Generator {
		Result.Status apply(CdsModel model) throws IOException;
	}

	static boolean noRelevantIssues(ParserMode parserMode, GeneratorMode generatorMode) {
		if (IssueCollectorFactory.hasIssues(IssueType.CRITICAL)) {
			return false;
		}
		if (parserMode == ParserMode.STRICT && IssueCollectorFactory.hasIssues(IssueType.UNRECOGNIZED)) {
			return false;
		}
		if (generatorMode == GeneratorMode.STRICT) {
			if (IssueCollectorFactory.hasIssues(IssueType.ERROR)) {
				return false;
			}
			return !IssueCollectorFactory.hasIssues(IssueType.UNSUPPORTED);
		}
		return true;
	}
}
