/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.google.common.base.Strings;
import com.sap.cds.generator.util.GeneratorMode;
import com.sap.cds.generator.util.ParserMode;

/**
 * Configuration instance stores the configuration values for the application.
 * There should exist only one instance per application. Normally should be
 * filled from the command line options. But for the tests and integration one
 * can create an empty configuration and use default values.
 *
 */
public class ConfigurationImpl implements Configuration {

	private ParserMode parserMode = ParserMode.TOLERANT;
	private GeneratorMode generatorMode = GeneratorMode.TOLERANT;
	private MethodStyle methodStyle = MethodStyle.BEAN;
	private String classNameSuffix = "_";
	private String basePackage = "";
	private boolean eventContext = false;
	private List<String> excludes = new ArrayList<>();
	private List<String> includes = new ArrayList<>();
	private boolean docs = true;
	private boolean fkAccessors = false;
	private GeneratedAnnotationDetailLevel generatedAnnotationDetailLevel = GeneratedAnnotationDetailLevel.FULL;
	private String pluginInformation = "";
	private boolean cqnServices = false;
	private boolean sharedInterfaces = false;
	private boolean uniqueEventContexts = false;
	private boolean interfacesForAspects = false;
	private String handlerPackageName; // This is set by mode in the generate goal and it intentionally has no default
	private boolean strictSetters = false;

	@Override
	public GeneratorMode getGeneratorMode() {
		return generatorMode;
	}

	public void setGeneratorMode(GeneratorMode generatorMode) {
		this.generatorMode = generatorMode != null ? generatorMode : GeneratorMode.TOLERANT;
	}

	@Override
	public String getClassNameSuffix() {
		return classNameSuffix;
	}

	@Override
	public ParserMode getParserMode() {
		return parserMode;
	}

	public void setParserMode(ParserMode parserMode) {
		this.parserMode = parserMode != null ? parserMode : ParserMode.TOLERANT;
	}

	@Override
	public MethodStyle getMethodStyle() {
		return methodStyle;
	}

	public void setMethodStyle(MethodStyle methodMode) {
		this.methodStyle = methodMode;
	}

	@Override
	public String getBasePackage() {
		return this.basePackage;
	}

	public void setBasePackage(String basePackage) {
		if (!Strings.isNullOrEmpty(basePackage)) {
			this.basePackage = basePackage;
		}
	}

	@Override
	public List<String> getExcludes() {
		return Collections.unmodifiableList(excludes);
	}

	public void setExcludes(List<String> excludes) {
		if (excludes != null) {
			this.excludes.addAll(excludes);
		}
	}

	@Override
	public boolean getEventContext() {
		return this.eventContext;
	}

	public void setEventContext(boolean eventContext) {
		this.eventContext = eventContext;
	}

	@Override
	public boolean getDocs() {
		return this.docs;
	}

	public void setDocs(boolean readDocs) {
		this.docs = readDocs;
	}

	@Override
	public List<String> getIncludes() {
		return Collections.unmodifiableList(includes);
	}

	public void setIncludes(List<String> includes) {
		if (includes != null) {
			this.includes.addAll(includes);
		}
	}

	@Override
	public boolean fkAccessors() {
		return this.fkAccessors;
	}

	public void setFkAccessors(boolean fkAccessors) {
		this.fkAccessors = fkAccessors;
	}

	@Override
	public String getPluginInformation() {
		return pluginInformation;
	}

	public void setPluginInformation(String pluginInformation) {
		this.pluginInformation = pluginInformation;
	}

	@Override
	public GeneratedAnnotationDetailLevel getGenerationAnnotationDetailLevel() {
		return this.generatedAnnotationDetailLevel;
	}

	public void setGenerationAnnotationDetailLevel(GeneratedAnnotationDetailLevel detailLevel) {
		this.generatedAnnotationDetailLevel = detailLevel;
	}

	@Override
	public boolean getCqnServices() {
		return this.cqnServices;
	}

	public void setCqnServices(boolean cqnServices) {
		this.cqnServices = cqnServices;
	}

	@Override
	public boolean getSharedInterfaces() {
		return sharedInterfaces;
	}

	public void setSharedInterfaces(boolean value) {
		sharedInterfaces = value;
	}

	@Override
	public boolean getUniqueEventContexts() {
		return uniqueEventContexts;
	}

	public void setUniqueEventContexts(boolean uniqueEventContexts) {
		this.uniqueEventContexts = uniqueEventContexts;
	}

	@Override
	public boolean getInterfacesForAspects() {
		return interfacesForAspects;
	}

	public void setInterfacesForAspects(boolean value) {
		this.interfacesForAspects = value;
	}

	@Override
	public String getHandlerPackageName() {
		return handlerPackageName;
	}

	public void setHandlerPackageName(String packageName) {
		handlerPackageName = packageName;
	}

	@Override
	public boolean getStrictSetters() {
		return strictSetters;
	}

	public void setStrictSetters(boolean strictSetters) {
		this.strictSetters = strictSetters;
	}

}
