/************************************************************************
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.util;

import static com.sap.cds.generator.util.NamesUtils.className;
import static com.sap.cds.generator.util.NamesUtils.eventContextClassName;
import static com.sap.cds.generator.writer.Types.RETURN_TYPE;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;
import javax.tools.JavaFileObject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.writer.Types;
import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsOperation;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.impl.CdsAnnotatableImpl;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.OnConditionAnalyzer;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class TypeUtils {

	private static final Logger logger = LoggerFactory.getLogger(TypeUtils.class);
	private static final ParameterizedTypeName MAP_STR2OBJ = ParameterizedTypeName.get(Types.MAP,
			Types.STRING, Types.OBJECT);

	private TypeUtils() {
	}

	public static TypeName getAttributeType(ClassName parent, CdsType type, Configuration cfg) {
		return getAttributeType(parent, type, cfg, null);
	}

	private static TypeName getAttributeType(ClassName parent, CdsType type, Configuration cfg, CdsElement element) {
		TypeName attributeType = null;

		if (type.isAssociation()) {
			CdsAssociationType assocType = type.as(CdsAssociationType.class);
			CdsType target = null;
			if (cfg.getInterfacesForAspects()) {
				// Target type first, then aspect
				if (assocType.getTarget() != null) {
					target = assocType.getTarget();
				} else {
					target = assocType.getTargetAspect().orElse(null);
				}
			} else if (!cfg.getInterfacesForAspects()) {
				// Aspect, then target type or nothing if there is none
				target = assocType.getTargetAspect().orElse(assocType.getTarget());
			}
			if (target != null) {
				attributeType = className(cfg, target);
				if (!CdsModelUtils.isSingleValued(type)) {
					attributeType = listOf(attributeType);
				}
			}
		} else if (type.isSimple()) {
			CdsSimpleType simpleType = type.as(CdsSimpleType.class);
			attributeType = TypeName.get(simpleType.getJavaType());
		} else if (type.isStructured()) {
			attributeType = type.getQualifiedName().isEmpty() ? null : className(cfg, type);
		} else if (type.isArrayed()) {
			// Two ways to fetch the type:
			if (element != null) {
				// inline types of an elements have names including the element name at the end
				attributeType = arrayedTypeNameForElement(parent, type.as(CdsArrayedType.class), cfg, element);
			} else {
				// others are just empty or have global type names
				attributeType = arrayedTypeNameForParameter(type.as(CdsArrayedType.class), cfg);
			}
		} else {
			logger.warn("Interface Generation: Unsupported CDS Element with attribute name '{}' and type '{}'.",
					type.getName(), type);
			return null;
		}
		return attributeType;
	}

	public static TypeName getReturnType(ClassName parent, CdsElement attribute, Configuration cfg) {
		boolean isMedia = attribute.annotations().anyMatch(a -> "Core.MediaType".equals(a.getName()));
		if (isMedia && attribute.getType().isSimple()) {
			CdsSimpleType simpleType = attribute.getType().as(CdsSimpleType.class);
			return ClassName.get(CdsBaseType.cdsJavaMediaType(simpleType.getType()));
		}
		if (attribute.getType().isStructured() && attribute.getType().getQualifiedName().isEmpty()) {
			return className(parent, attribute);
		} else if (isAnonymousAspect(attribute)) {
			TypeName attributeType = className(parent, attribute);
			if (!CdsModelUtils.isSingleValued(attribute.getType())) {
				attributeType = listOf(attributeType);
			}
			return attributeType;
		}
		return getAttributeType(parent, attribute.getType(), cfg, attribute);
	}

	public static TypeName getOperationResultType(CdsEntity boundTo, CdsOperation operation, CdsType returnType, Configuration config) {
		if (TypeUtils.isAnonymousType(returnType, config)) {
			ClassName innerEventContextClassName = eventContextClassName(config, boundTo, operation)
				.nestedClass(RETURN_TYPE);
			if (returnType.isArrayed()) {
				return getArrayTypeName(innerEventContextClassName);
			}
			return innerEventContextClassName;
		}

		if (returnType.isStructured()) {
			return className(config, returnType);
		} else if (returnType.isSimple()) {
			CdsSimpleType type = returnType.as(CdsSimpleType.class);
			return TypeName.get(type.getJavaType());
		} else if (returnType.isArrayed()) {
			if (config.getSharedInterfaces() && TypeUtils.isAnonymousType(returnType)) {
				ClassName className = className(config, returnType);
				return getAttributeType(className, returnType, config, null);
			} else {
				TypeName nestedType = getOperationResultType(boundTo, operation, returnType.as(CdsArrayedType.class).getItemsType(), config);
				return ParameterizedTypeName.get(Types.COLLECTION, nestedType);
			}
		} else {
			logger.warn("Consumption Interface Generation: Unsupported CDS Element with attribute name {} and type {}",
					operation.getName(), returnType.getName());
			return null;
		}
	}

	public static ParameterizedTypeName listOf(TypeName type) {
		return ParameterizedTypeName.get(Types.LIST, type);
	}


	public static boolean isAnonymousAspect(CdsElement element) {
		CdsType type = element.getType();
		if (type.isAssociation()) {
			Optional<CdsStructuredType> targetAspect = type.as(CdsAssociationType.class).getTargetAspect();
			if (targetAspect.isPresent() && targetAspect.get().getQualifiedName().isEmpty()) {
				return true;
			}
		}
		return false;
	}

	public static boolean isAnonymousType(CdsType type, Configuration config) {
		if (type.isArrayed()) {
			boolean isTypeAnonymous = TypeUtils.isAnonymousType(type);
			if (config.getSharedInterfaces()) {
				return isTypeAnonymous && type.getQualifiedName().isEmpty();
			} else {
				return isTypeAnonymous;
			}
		}
		return isAnonymousType(type);
	}

	public static boolean isAnonymousType(CdsType type) {
		if (type == null) {
			return false;
		} else if (type.isArrayed()) {
			return isAnonymousType(type.as(CdsArrayedType.class).getItemsType());
		} else if (type.isStructured() && type.getQualifiedName().isEmpty()) {
			return true;
		}
		return false;
	}

	public static Stream<CdsElement> getAnonymousElements(CdsElement element) {
		CdsType type = element.getType();
		if (type.isAssociation()) {
			Optional<CdsStructuredType> targetAspect = type.as(CdsAssociationType.class).getTargetAspect();
			if (targetAspect.isPresent() && targetAspect.get().getQualifiedName().isEmpty()) {
				return targetAspect.get().as(CdsStructuredType.class).elements();
			}
		}
		return Stream.empty();
	}

	public static void addStaticFactoryMethods(TypeSpec.Builder builder, TypeName returnType,
			TypeSpec.Builder... anonymousbuilder) {

		MethodSpec.Builder createMethodBuilder = MethodSpec.methodBuilder("create").returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);
		CodeBlock.Builder createCodeBuilder = CodeBlock.builder();
		createCodeBuilder.addStatement("return $T.create($T.class)", Types.STRUCT, returnType);
		createMethodBuilder.addCode(createCodeBuilder.build());

		MethodSpec.Builder ofMethodBuilder = MethodSpec.methodBuilder("of").returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC)
				.addParameter(ParameterSpec.builder(MAP_STR2OBJ, "map").build());
		CodeBlock.Builder ofCodeBuilder = CodeBlock.builder();
		ofCodeBuilder.addStatement("return $T.access(map).as($T.class)", Types.STRUCT, returnType);
		ofMethodBuilder.addCode(ofCodeBuilder.build());

		if (anonymousbuilder.length > 0) {
			anonymousbuilder[0].addMethod(createMethodBuilder.build());
			anonymousbuilder[0].addMethod(ofMethodBuilder.build());
		} else {
			builder.addMethod(createMethodBuilder.build());
			builder.addMethod(ofMethodBuilder.build());
		}
	}

	public static void addStaticCreateForKeys(TypeSpec.Builder builder, ClassName entity, Map<String, ParameterSpec> parameters) {
		if (!parameters.isEmpty()) {
			MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("create").returns(entity)
														 .addModifiers(Modifier.PUBLIC, Modifier.STATIC);
			parameters.forEach((name, spec) -> methodBuilder.addParameter(spec));

			CodeBlock.Builder codeBuilder = CodeBlock.builder();
			codeBuilder.addStatement("$T<String, Object> keys = new $T<>()", Types.MAP, Types.HASH_MAP);
			parameters.forEach((name, spec) ->
					codeBuilder.addStatement("keys.put($L, $L)", name, spec.name));
			codeBuilder.addStatement("return $T.access(keys).as($T.class)", Types.STRUCT, entity);

			methodBuilder.addCode(codeBuilder.build());
			builder.addMethod(methodBuilder.build());
		}
	}


	public static List<CdsElement> getManagedToOneFks(CdsElement attribute) {
		return new OnConditionAnalyzer(attribute, false).getFkMapping().keySet().stream()
				.map(m -> attribute.getDeclaringType().as(CdsStructuredType.class).findElement(m)).map(Optional::get)
				.toList();
	}

	public static void logWarningForManyToManyWithStructElement(CdsModel model, CdsStructuredType struct) {
		if (struct.elements().anyMatch(e -> e.getType().isStructured())) {
			Optional<CdsElement> assocElement = model.entities().flatMap(CdsStructuredType::associations)
					.filter(a -> a.getType().as(CdsAssociationType.class).getTargetAspect()
							.map(ta -> ta.getName().equals(struct.getName())).orElse(false))
					.findFirst();
			if (assocElement.isPresent()) {
				logger.warn(
						"Limited CRUD operation support available with composition of aspects({}) containing structured elements. Use dynamic queries in such cases.",
						struct.getQualifiedName());
			}
		}
	}

	private static TypeName arrayedTypeNameForParameter(CdsArrayedType type, Configuration cfg) {
		CdsType itemsType = type.getItemsType();
		if (itemsType.isSimple()) {
			// Erase the type (global arrayed or global item type if the item type is simple to remain compatible)
			return getArrayTypeName(TypeName.get(itemsType.as(CdsSimpleType.class).getJavaType()));
		} else {
			ClassName javaClassName = null;
			if (!type.getQualifiedName().isEmpty() && itemsType.getQualifiedName().isEmpty()) {
				// The arrayed type itself generated as Collection.Item
				return getArrayTypeName(className(cfg, type).nestedClass(NamesUtils.ITEM_TYPE_NAME));
			} else if (!itemsType.getQualifiedName().isEmpty()) {
				javaClassName = className(cfg, itemsType);
			}

			if (javaClassName == null) {
				logger.error("Interface Generation: Unsupported combination of empty type and items type for function/action parameter.");
				return null;
			}

			return getArrayTypeName(javaClassName);
		}
	}

	private static TypeName arrayedTypeNameForElement(ClassName parent, CdsArrayedType type, Configuration cfg, CdsElement element) {
		CdsType itemsType = type.getItemsType();
		TypeName innerAttributeType = getAttributeType(parent, itemsType, cfg, element);
		if (innerAttributeType == null) {
			if (type.getQualifiedName().startsWith(element.getDeclaringType().getQualifiedName())) {
				return getArrayTypeName(className(parent, element));
			} else {
				return getArrayTypeName(className(cfg, type).nestedClass(NamesUtils.ITEM_TYPE_NAME));
			}
		} else {
			return getArrayTypeName(innerAttributeType);
		}
	}

	public static TypeName getArrayTypeName(TypeName javaType) {
		return ParameterizedTypeName.get(Types.COLLECTION, javaType);
	}

	public static void writeType(String packageName, TypeSpec typeSpec, GeneratedFile.Consumer consumer) {
		final JavaFile javaFile = JavaFile.builder(packageName, typeSpec).build();
		final JavaFileObject fileObject = javaFile.toJavaFileObject();
		GeneratedFile generatedFile = new GeneratedFile() {
			@Override
			public URI getUri() {
				return fileObject.toUri();
			}

			@Override
			public InputStream getContent() throws IOException {
				return fileObject.openInputStream();
			}
		};
		try {
			consumer.accept(generatedFile);
		} catch (IOException e) {
			String message = "Exception while writing to file %s.".formatted(generatedFile.getUri());
			throw new EntityWriterException(message, e);
		}
	}

	/**
	 * Checks whether the given {@link CdsAnnotatable} is to be ignored by the code generation.
	 *
	 * @param def the cds definition
	 * @return true if to be ignored by the code gen, false otherwise
	 */
	public static boolean isIgnored(CdsAnnotatable def) {
		String cdsJavaIgnore = CdsAnnotatableImpl.removeAt(CdsConstants.ANNOTATION_CDS_JAVA_IGNORE);
		return def.getAnnotationValue(cdsJavaIgnore, false);
	}
}
