/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.CaseFormatHelper.toUpperUnderscore;
import static com.sap.cds.generator.util.NamesUtils.argumentName;
import static com.sap.cds.generator.util.NamesUtils.warnOnJavaKeywords;
import static com.sap.cds.generator.util.NamesUtils.className;
import static com.sap.cds.generator.util.NamesUtils.eventContextClassName;
import static com.sap.cds.generator.util.NamesUtils.methodName;
import static com.sap.cds.generator.util.NamesUtils.suffixedClassName;
import static com.sap.cds.generator.util.NamesUtils.typedServiceBuilderName;
import static com.sap.cds.generator.util.TypeUtils.getArrayTypeName;
import static com.sap.cds.generator.util.TypeUtils.getAttributeType;
import static com.sap.cds.generator.util.TypeUtils.getOperationResultType;
import static com.sap.cds.generator.util.TypeUtils.isAnonymousType;
import static com.sap.cds.generator.writer.SpecWriterUtil.cdsNameAnnotation;
import static com.sap.cds.generator.writer.Types.DOT_CDS_NAME;

import java.util.List;
import java.util.Optional;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.generator.util.TypeUtils;
import com.sap.cds.generator.writer.ModelWriter.Context;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsBoundAction;
import com.sap.cds.reflect.CdsBoundFunction;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsOperation;
import com.sap.cds.reflect.CdsParameter;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class CreateAppServiceInterfaceVisitor implements CdsVisitor {

	private static final String DRAFT_SUFFIX = "_drafts";
	private static final ClassName INNER_DRAFT = ClassName.get("", "Draft");
	private static final ClassName INNER_APPLICATION = ClassName.get("", "Application");
	private static final ClassName INNER_REMOTE = ClassName.get("", "Remote");
	private final TypeSpec.Builder builder;
	private final Configuration config;
	private final NamesUtils namesUtils;
	private final ClassName className;

	CreateAppServiceInterfaceVisitor(TypeSpec.Builder builder, ClassName className, Context context) {
		this.builder = builder;
		this.className = className;
		this.config = context.config();
		this.namesUtils = context.namesUtils();
	}

	@Override
	public void visit(CdsService service) {
		if (namesUtils.isExcluded(service.getQualifiedName()) || TypeUtils.isIgnored(service)) {
			return;
		}

		warnOnJavaKeywords(service.getQualifiedName());

		builder.addSuperinterface(Types.CQN_SERVICE);

		// add annotation @CdsName(<BuilderClassName>.CDS_NAME) at service interface
		builder.addAnnotation(cdsNameAnnotation(typedServiceBuilderName(config, service), "$T.CDS_NAME"));

		addInnerInterface(INNER_APPLICATION, new ClassName[] { Types.APPLICATION_SERVICE, className});
		addInnerInterface(INNER_REMOTE, new ClassName[] { Types.REMOTE_SERVICE, className});
		if (isServiceDraftEnabled(service)) {
			addInnerInterface(INNER_DRAFT, new ClassName[] { Types.DRAFT_SERVICE, className});
		}

		service.entities().forEach(entity -> entity.accept(this));
		service.actions().forEach(action -> action.accept(this));
		service.functions().forEach(function -> function.accept(this));
	}

	private void addInnerInterface(ClassName innerInterfaceName, ClassName[] superInterfaces) {
		TypeSpec.Builder innerInterfaceBuilder = TypeSpec.interfaceBuilder(innerInterfaceName)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);
		List.of(superInterfaces).forEach(innerInterfaceBuilder::addSuperinterface);
		builder.addType(innerInterfaceBuilder.build());
	}

	@Override
	public void visit(CdsEntity entity) {
		if (entity.getName().endsWith(DRAFT_SUFFIX) || namesUtils.isExcluded(entity.getQualifiedName())
				|| TypeUtils.isIgnored(entity)) {
			return;
		}

		warnOnJavaKeywords(entity.getQualifiedName());

		entity.actions().forEach(action -> {
			// don't create methods for draft actions
			if (CdsConstants.DRAFT_ACTIONS.contains(action.getName())) {
				return;
			}

			Optional<MethodSpec> method = buildOperationFacadeMethod(action, action.returnType(), Optional.of(entity));
			method.ifPresent(builder::addMethod);
		});

		entity.functions().forEach(function -> {
			Optional<MethodSpec> method = buildOperationFacadeMethod(function, function.returnType(),
					Optional.of(entity));
			method.ifPresent(builder::addMethod);
		});
	}

	@Override
	public void visit(CdsFunction function) {
		if (namesUtils.isExcluded(function.getQualifiedName()) || TypeUtils.isIgnored(function)) {
			return;
		}

		Optional<MethodSpec> method = buildOperationFacadeMethod(function, function.returnType(), Optional.empty());
		method.ifPresent(builder::addMethod);
	}

	@Override
	public void visit(CdsAction action) {
		if (namesUtils.isExcluded(action.getQualifiedName()) || TypeUtils.isIgnored(action)) {
			return;
		}

		Optional<MethodSpec> method = buildOperationFacadeMethod(action, action.returnType(), Optional.empty());
		method.ifPresent(builder::addMethod);
	}

	private Optional<MethodSpec> buildOperationFacadeMethod(CdsOperation operation, Optional<CdsType> cdsReturnType,
			Optional<CdsEntity> entityOpt) {
		warnOnJavaKeywords(operation.getQualifiedName());

		if (TypeUtils.isIgnored(operation)) {
			return Optional.empty();
		}

		String methodName = methodName(operation);
		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(methodName)
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);

		ClassName eventContextClassName = eventContextClassName(config, entityOpt.orElse(null), operation);
		// add annotation @CdsName(<EventContextClassName>.CDS_NAME)
		methodBuilder.addAnnotation(cdsNameAnnotation(eventContextClassName.simpleName() + DOT_CDS_NAME, "$L"));

		if (cdsReturnType.isPresent()) {
			TypeName returnType = getOperationResultType(entityOpt.orElse(null), operation, cdsReturnType.get(), config);
			methodBuilder.returns(returnType);
		}

		if (entityOpt.isPresent()) {
			CdsEntity entity = entityOpt.get();
			CdsParameter bindingParameter = null;
			if (operation instanceof CdsBoundAction action) {
				bindingParameter = action.getBindingParameter();
			} else if (operation instanceof CdsBoundFunction function) {
				bindingParameter = function.getBindingParameter();
			}

			String bindingParameterName = bindingParameter != null ? bindingParameter.getName() : "ref";

			TypeName builderType = suffixedClassName(config, entity);

			ParameterSpec bindingParam = ParameterSpec.builder(builderType, bindingParameterName).build();
			methodBuilder.addParameter(bindingParam);
		}

		operation.parameters().filter(p -> !TypeUtils.isIgnored(p)).forEach(param -> {
			TypeName paramType = getParameterType(entityOpt.orElse(null), operation, param);

			// creates a literal pointing to a constant in the event context interface, e.g. AddToOrderContext.ORDER_ID
			String cdsNameValue = eventContextClassName.simpleName() + "." + toUpperUnderscore(param.getName());
			// creates a Java parameter name with lower camel case, e.g. "order_ID" -> "orderId"
			ParameterSpec paramSpec = ParameterSpec.builder(paramType, argumentName(param))
				.addAnnotation(cdsNameAnnotation(cdsNameValue, "$L")).build();
			methodBuilder.addParameter(paramSpec);
		});

		return Optional.of(methodBuilder.build());
	}

	private TypeName getParameterType(CdsEntity boundTo, CdsOperation operation, CdsParameter param) {
		// If the type is anonymous (no global name or the configuration does not let use the global name)
		// the type of the parameter is the name of the interface defined within the operation context
		// Otherwise, the generic type name is used (also could be anonymous or even primitive type name)
		if (isAnonymousType(param.getType(), config)) {
			TypeName innerType = className(eventContextClassName(config, boundTo, operation), param);
			if (param.getType().isArrayed()) {
				return getArrayTypeName(innerType);
			} else {
				return innerType;
			}
		} else {
			return getAttributeType(null, param.getType(), config);
		}
	}

	private static boolean isServiceDraftEnabled(CdsService service) {
		return service.entities()
				.anyMatch(entity -> entity.findAnnotation(CdsConstants.ANNOTATION_DRAFT_ENABLED).isPresent());
	}
}
