/*******************************************************************
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.generator.writer;

import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.util.GeneratedFile;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.generator.util.TypeUtils;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsOperation;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.TypeSpec;

public class CreateTemplateHandlerClassVisitor implements CdsVisitor {

	private final Configuration configuration;
	private final GeneratedFile.Consumer consumer;
	private final NamesUtils namesUtils;
	private final CdsModel model;

	public CreateTemplateHandlerClassVisitor(Configuration configuration, CdsModel model, GeneratedFile.Consumer consumer) {
		this.configuration = configuration;
		this.consumer = consumer;
		this.namesUtils = new NamesUtils(configuration);
		this.model = model;
	}

	@Override
	public void visit(CdsEntity entity) {
		if (!TypeUtils.isIgnored(entity)) {
			Stream.concat(entity.functions(), entity.actions())
				.filter(o -> !CdsConstants.DRAFT_ACTIONS.contains(o.getName()))
				.filter(o -> !TypeUtils.isIgnored(o)
					// This supplies to isExcluded qualified name as if the operation is unbound
					// to make simple patterns like context.** or **.actionName yield same
					// results for both bound and unbound operations.
					&& !namesUtils.isExcluded(String.join(".", entity.getQualifiedName(), o.getQualifiedName())))
				.forEach(o -> generate(entity, o));
		}
	}

	@Override
	public void visit(CdsAction action) {
		if (isRelevant(action)) {
			generate(action);
		}
	}

	@Override
	public void visit(CdsFunction function) {
		if (isRelevant(function)) {
			generate(function);
		}
	}

	private void generate(CdsOperation operation) {
		ClassName eventContextClassName = NamesUtils.eventContextClassName(configuration, null, operation);
		ClassName handlerClassName = NamesUtils.templateEventHandlerClassName(configuration, operation);
		ClassName serviceWrapperClassName = NamesUtils.typedServiceBuilderName(configuration,
			model.getService(operation.getQualifier()));

		generateHandlerClass(operation, eventContextClassName, serviceWrapperClassName, handlerClassName,
			NamesUtils.getOnHandlerMethodName(operation));
	}

	private void generate(CdsEntity boundTo, CdsOperation operation) {
		ClassName eventContextClassName = NamesUtils.eventContextClassName(configuration, boundTo, operation);
		ClassName handlerClassName = NamesUtils.templateEventHandlerClassName(configuration, boundTo, operation);
		ClassName serviceWrapperClassName = NamesUtils.typedServiceBuilderName(configuration,
			model.getService(boundTo.getQualifier()));

		generateHandlerClass(operation, eventContextClassName, serviceWrapperClassName, handlerClassName,
			NamesUtils.getOnHandlerMethodName(operation));
	}

	private void generateHandlerClass(CdsOperation operation, ClassName eventContextClassName,
		ClassName serviceWrapperClassName, ClassName handlerName, String methodName) {

		TypeSpec.Builder builder = TypeSpec.classBuilder(handlerName)
			.addModifiers(Modifier.PUBLIC)
			// is a spring boot component
			.addAnnotation(Types.SPRING_BOOT_COMPONENT)
			// Implements EventHandler so is visible to CAP runtime
			.addSuperinterface(Types.EVENT_HANDLER);

		// Bind the handler to the qualifier service of the operation
		AnnotationSpec.Builder serviceName = AnnotationSpec.builder(Types.SERVICE_NAME);

		serviceName.addMember("value", "$T.CDS_NAME", serviceWrapperClassName);
		builder.addAnnotation(serviceName.build());

		// Documentation
		if (configuration.getDocs()) {
			operation.getDoc().ifPresent(builder::addJavadoc);
		}

		// @On handler by default
		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(methodName)
			.addModifiers(Modifier.PUBLIC)
			.addParameter(ParameterSpec.builder(eventContextClassName, "context").build())
			//add @On annotation for CAP Java event handling
			.addAnnotation(Types.HANDLER_ON)
			.addComment("Your code goes here")
			.addStatement("$N.setCompleted()", "context");

		builder.addMethod(methodBuilder.build());

		TypeSpec typeSpec = builder.build();

		TypeUtils.writeType(configuration.getHandlerPackageName(), typeSpec, consumer);
	}

	private boolean isRelevant(CdsDefinition definition) {
		return !(TypeUtils.isIgnored(definition) || namesUtils.isExcluded(definition.getQualifiedName()));
	}

}
