/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.writer;

import static com.google.common.collect.Streams.concat;
import static com.sap.cds.generator.util.NamesUtils.className;
import static com.sap.cds.generator.util.NamesUtils.getResolvedWrapperName;
import static com.sap.cds.generator.util.NamesUtils.isValidTechnicalEntity;
import static com.sap.cds.generator.util.NamesUtils.qualifiedContextName;
import static com.sap.cds.generator.util.NamesUtils.qualifiedWrapperBuilderName;
import static com.sap.cds.generator.util.NamesUtils.suffixedClassName;
import static com.sap.cds.generator.util.NamesUtils.unqualifiedName;
import static com.sap.cds.generator.util.NamesUtils.warnOnJavaKeywords;
import static com.sap.cds.generator.util.TypeUtils.logWarningForManyToManyWithStructElement;
import static com.sap.cds.generator.util.TypeUtils.writeType;
import static com.sap.cds.generator.writer.SpecWriterUtil.cdsNameAnnotation;
import static com.sap.cds.generator.writer.SpecWriterUtil.getJavaDoc;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.BooleanSupplier;
import java.util.stream.Collectors;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.util.GeneratedAnnotationUtil;
import com.sap.cds.generator.util.GeneratedFile.Consumer;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.generator.util.TypeUtils;
import com.sap.cds.generator.util.UnSupportedModelException;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsEnumType;
import com.sap.cds.reflect.CdsEvent;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsKind;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.reflect.impl.CdsEventBuilder.EventProxy;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.TypeSpec.Builder;

public class ModelWriter implements CdsVisitor {

	private final Consumer consumer;
	private final Configuration config;
	private final CdsModel model;
	private final Map<String, CdsDefinition> wrapperMap;
	private final Set<String> builderInterfaces;
	private final NamesUtils namesUtils;
	private final GeneratedAnnotationUtil generated;
	private final List<String> generatedServices = new ArrayList<>();
	private final Context context;

	public ModelWriter(Consumer consumer, Configuration config, CdsModel model) {
		this.consumer = consumer;
		this.config = config;
		this.model = model;
		this.wrapperMap = new HashMap<>();
		this.builderInterfaces = new HashSet<>();
		this.namesUtils = new NamesUtils(config);
		this.generated = new GeneratedAnnotationUtil(config);
		this.context = new Context(model.getMeta(CdsConstants.META_TENANT_DISCRIMINATOR), this.config, this.namesUtils);
	}

	@Override
	public void visit(CdsModel model) {
		wrapperMap.values().forEach(this::generateWrapperInterface);
	}

	@Override
	public void visit(CdsEntity entity) {
		warnOnJavaKeywords(entity.getQualifiedName());
		if (!namesUtils.isExcluded(entity.getQualifiedName())
				&& isValidTechnicalEntity(config, model, entity)) {
			collectWrapperInterfaces(entity);
			generateBuilderInterface(entity);
			generateConsumptionInterface(entity);
			if (config.getEventContext()) {
				concat(entity.actions(), entity.functions()).forEach(a -> generateEventContextInterface(a, entity));
			}
		}
	}

	@Override
	public void visit(CdsEvent event) {
		warnOnJavaKeywords(event.getQualifiedName());
		if (!namesUtils.isExcluded(event.getQualifiedName())) {
			collectWrapperInterfaces(event);
			generateBuilderInterface(event);
			generateConsumptionInterface(event);
			if (config.getEventContext()) {
				generateEventContextInterface(event, null);
			}
		}
	}

	@Override
	public void visit(CdsAction action) {
		if (config.getEventContext() && !namesUtils.isExcluded(action.getQualifiedName())) {
			collectWrapperInterfaces(action);
			generateEventContextInterface(action, null);
		}
	}

	@Override
	public void visit(CdsFunction function) {
		if (config.getEventContext() && !namesUtils.isExcluded(function.getQualifiedName())) {
			collectWrapperInterfaces(function);
			generateEventContextInterface(function, null);
		}
	}

	@Override
	public void visit(CdsArrayedType type) {
		// Explicit types for Arrayed types are required only if the inner type is
		// anonymous and not simple.
		// Explicit inner type e.g. many Books will be represented as Collection<Books>
		// Simple inner type will be represented as Collection<Integer>
		if (!type.getQualifiedName().isEmpty() && type.getItemsType().isStructured()
				&& (type.getItemsType().getQualifiedName().isEmpty() && !type.getItemsType().isSimple())) {

			warnOnJavaKeywords(type.getQualifiedName());
			if (!namesUtils.isExcluded(type.getQualifiedName())) {
				generateConsumptionInterface(type);
			}
		}
	}

	@Override
	public void visit(CdsStructuredType struct) {
		if (!struct.getName().isEmpty()) {
			warnOnJavaKeywords(struct.getQualifiedName());
			if (!namesUtils.isExcluded(struct.getQualifiedName())) {
				generateConsumptionInterface(struct);
				generateBuilderInterface(struct);
				if (isTargetAspect(struct)) {
					// For every TargetAspect which contains structured elements and is target of
					// many to many composition, throw warning
					logWarningForManyToManyWithStructElement(model, struct);
				}
			}
		}
	}

	@Override
	public void visit(CdsService service) {
		generateTypedAppServiceInterface(service);
	}

	@Override
	public void visit(CdsEnumType<?> type) {
		if (!namesUtils.isExcluded(type.getQualifiedName()) && !TypeUtils.isIgnored(type)) {
			TypeSpec.Builder result = TypeSpec.classBuilder(className(config, type))
				.addModifiers(Modifier.PUBLIC, Modifier.FINAL);
			result.addAnnotation(cdsNameAnnotation(type.getQualifiedName(), "$S"));
			type.accept(new CreateEnumConstantClassVisitor(result));
			if (!result.fieldSpecs.isEmpty()) {
				generated.addTo(result);
				writeType(NamesUtils.packageName(type, config.getBasePackage()), result.build(), consumer);
			}
		}
	}

	public List<String> getGeneratedServices() {
		return this.generatedServices;
	}

	private void collectWrapperInterfaces(CdsDefinition def) {
		String packageName = NamesUtils.packageName(def, config.getBasePackage());
		wrapperMap.put(packageName, def);
	}

	private void generateWrapperInterface(CdsDefinition def) {
		String qualifiedName = def.getQualifiedName();
		String name = def.getName();
		String packageName = NamesUtils.packageName(def, config.getBasePackage());
		String qualifiedWrapperName = qualifiedWrapperBuilderName(def, config.getClassNameSuffix(), true);
		while (builderInterfaces.contains(qualifiedWrapperName.toLowerCase(Locale.US))) {
			qualifiedWrapperName = getResolvedWrapperName(qualifiedWrapperName, config.getClassNameSuffix());
		}

		Builder builder = TypeSpec.interfaceBuilder(unqualifiedName(qualifiedWrapperName))
				.addModifiers(Modifier.PUBLIC);

		generated.addTo(builder);

		if (new WrapperInterfaceCreator(builder, model, config)
				.generateInterface(qualifiedContextName(qualifiedName, name))) {
			writeType(packageName, builder.build(), consumer);
		}
	}

	private void generateConsumptionInterface(CdsType def) {
		ClassName className = className(config, def);
		Builder builder = createInterfaceSpecBuilder(className);
		addJavadoc(builder, def, () -> !(def instanceof EventProxy) && config.getDocs());

		// Explicit "superclasses" via annotation. Assume qualified CDS names in the annotation.
		// Assume also that we do not need to traverse the types unless there will be the support
		// for the proper inheritance in CDS
		Collection<CdsStructuredType> parents =
			def.getAnnotationValue(CdsConstants.ANNOTATION_JAVA_EXTENDS, Collections.<String>emptyList())
				.stream().map(model::getStructuredType).map(this::throwErrorOnExtendEntity)
				.collect(Collectors.toSet());
		def.accept(new CreateConsumptionInterfaceVisitor(builder, className, context, parents));

		def.getAnnotationValue(CdsConstants.ANNOTATION_JAVA_EXTENDS.substring(1), Collections.<String>emptyList())
			.stream()
				.map(model::getStructuredType)
				.map(this::throwErrorOnExtendEntity)
				.map(d -> className(config, d))
				.forEach(builder::addSuperinterface);

		generated.addTo(builder);

		TypeSpec typeSpec = builder.build();
		String packageName = NamesUtils.packageName(def, config.getBasePackage());
		writeType(packageName, typeSpec, consumer);
	}

	private void generateEventContextInterface(CdsDefinition def, CdsEntity boundEntity) {
		if (TypeUtils.isIgnored(def)) {
			return;
		}

		ClassName interfaceName = NamesUtils.eventContextClassName(config, boundEntity, def);
		Builder builder = createInterfaceSpecBuilder(interfaceName);
		addJavadoc(builder, def, config::getDocs);

		String boundEntityName = boundEntity == null ? null : boundEntity.getQualifiedName();

		def.accept(new CreateEventContextInterfaceVisitor(builder, interfaceName, boundEntityName, context));

		generated.addTo(builder);

		TypeSpec typeSpec = builder.build();

		writeType(interfaceName.packageName(), typeSpec, consumer);
	}

	private CdsStructuredType throwErrorOnExtendEntity(CdsStructuredType struct) {
		if (struct.getKind().equals(CdsKind.ENTITY) && !struct.as(CdsEntity.class).isAbstract()) {
			throw new UnSupportedModelException(
					"The '@cds.java.extends' annotation does not support extending an entity.");
		}
		return struct;
	}

	private void generateBuilderInterface(CdsType def) {
		ClassName builderClassName = suffixedClassName(config, def);
		ClassName entityClassName = className(config, def);

		Builder builder = createInterfaceSpecBuilder(builderClassName);
		addJavadoc(builder, def, config::getDocs);

		def.accept(new CreateBuilderInterfaceVisitor(builder, builderClassName, entityClassName, context));

		builderInterfaces
				.add(qualifiedWrapperBuilderName(def, config.getClassNameSuffix(), false).toLowerCase(Locale.US));

		generated.addTo(builder);

		TypeSpec typeSpec = builder.build();
		String packageName = NamesUtils.packageName(def, config.getBasePackage());
		writeType(packageName, typeSpec, consumer);
	}

	private void generateTypedAppServiceInterface(CdsService def) {
		// Classes generated by event context generation are required for action method
		// parameters and result type
		if (!config.getEventContext() || !config.getCqnServices() || namesUtils.isExcluded(def.getQualifiedName())
				|| TypeUtils.isIgnored(def)) {
			return;
		}

		ClassName interfaceName = NamesUtils.typedServiceClassName(config, def);
		Builder builder = TypeSpec.interfaceBuilder(interfaceName).addModifiers(Modifier.PUBLIC);

		addJavadoc(builder, def, config::getDocs);
		generated.addTo(builder);

		def.accept(new CreateAppServiceInterfaceVisitor(builder, interfaceName, context));

		TypeSpec typeSpec = builder.build();
		writeType(interfaceName.packageName(), typeSpec, consumer);
		this.generatedServices.add(interfaceName.canonicalName());
	}

	private static void addJavadoc(Builder builder, CdsDefinition def, BooleanSupplier doRead) {
		if (doRead.getAsBoolean()) {
			getJavaDoc(def).ifPresent(a -> builder.addJavadoc(a.replace("$", "$$")));
		}
	}

	private static Builder createInterfaceSpecBuilder(ClassName type) {
		return TypeSpec.interfaceBuilder(type).addModifiers(Modifier.PUBLIC);
	}

	private boolean isTargetAspect(CdsStructuredType struct) {
		return model.entities().flatMap(CdsStructuredType::associations)
				.map(a -> a.getType().as(CdsAssociationType.class).getTargetAspect())
				.anyMatch(ta -> ta.isPresent() && ta.get().getName().equals(struct.getName()));
	}

	/**
	 * A context to hold the tenant discriminator, configuration and utilities.
	 */
	record Context(String tenantDiscriminator, Configuration config, NamesUtils namesUtils) {

		/**
		 * Check if the given element is the tenant discriminator.
		 *
		 * @param element the {@link CdsElement} to check
		 * @return {@code true} if the given element is a tenant discriminator, {@code false} otherwise
		 */
		boolean isTenantDiscriminator(CdsElement element) {
			if (tenantDiscriminator == null || element == null) {
				return false;
			}
			return tenantDiscriminator.equals(element.getName());
		}
	}
}
