/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.CaseFormatHelper.lowercaseFirst;
import static com.sap.cds.generator.util.CaseFormatHelper.toUpperUnderscore;
import static com.sap.cds.generator.util.NamesUtils.constantName;
import static com.sap.cds.generator.util.NamesUtils.methodName;

import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.util.TypeUtils;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.CdsName;
import com.sap.cds.ql.CdsPath;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsType;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeSpec;

public class SpecWriterUtil {

	private SpecWriterUtil() {
		// avoid instances
	}

	public static boolean addCdsNameAnnotation(MethodSpec.Builder builder, CdsElement attribute) {
		if (!TypeUtils.isIgnored(attribute)) {
			String attributeName = propertyName(attribute.getName());

			boolean addCdsNameAnnotation = !attributeName.equals(methodName(attribute));
			if (addCdsNameAnnotation) {
				builder.addAnnotation(cdsNameAnnotation(constantName(attribute), "$L"));
			}
			return addCdsNameAnnotation;

		}
		return false;
	}

	public static boolean addCdsPathAnnotation(MethodSpec.Builder builder, String cdsPath, String javaName) {
		boolean addCdsPathAnnotation = !cdsPath.equals(propertyName(javaName));
		if (addCdsPathAnnotation) {
			builder.addAnnotation(cdsPathAnnotation(toUpperUnderscore(cdsPath), "$L"));
		}
		return addCdsPathAnnotation;
	}

	public static AnnotationSpec cdsNameAnnotation(Object cdsName, String format) {
		return AnnotationSpec.builder(CdsName.class).addMember("value", format, cdsName).build();
	}

	public static AnnotationSpec cdsPathAnnotation(String cdsName, String format) {
		return AnnotationSpec.builder(CdsPath.class).addMember("value", format, cdsName).build();
	}

	public static void addStaticField(TypeSpec.Builder builder, CdsElement element) {
		if (!TypeUtils.isIgnored(element)) {
			FieldSpec staticField = FieldSpec.builder(String.class, constantName(element))
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL).initializer("$S", element.getName())
				.build();
			builder.addField(staticField);
		}
	}

	public static Optional<String> getJavaDoc(CdsAnnotatable def) {
		return def.getDoc();
	}

	public static void setJavaDoc(CdsAnnotatable attribute, MethodSpec.Builder methodBuilder) {
		getJavaDoc(attribute).ifPresent(a -> methodBuilder.addJavadoc(a.replace("$", "$$")));
	}

	public static void addFkStaticField(TypeSpec.Builder builder, CdsElement attribute) {
		// Use value refs from getFkMapping method to construct the path to foreign key
		List<String> keys = resolveKeys(attribute)
				.map(r -> r.path())
				.toList();
		
		if (!keys.isEmpty()) {
			keys.forEach(fk -> {
				FieldSpec field = FieldSpec.builder(String.class, toUpperUnderscore(fk.replace(".", "_")))
						.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL)
						.initializer("$S", fk).build();
				builder.addField(field);
			});
		}
	}
	
	private static Stream<CqnReference> resolveKeys(CdsElement assoc) {
		return resolveKeys(CQL.get(assoc.getName()), assoc.getType());
	}

	private static Stream<CqnReference> resolveKeys(CqnReference path, CdsAssociationType assoc) {
		return assoc.refs().flatMap(r -> {
			CqnReference ref = CQL.get(path.path() + "." + r.path());
			CdsType type = assoc.getTarget().getElement(r.path()).getType();
			if (type instanceof CdsAssociationType a) {
				return resolveKeys(ref, a);
			}
			return Stream.of(ref);
		});
	}

	private static String propertyName(String javaName) {
		// Replica of same logic from ProxyCreator.
		if (javaName.length() > 3 && (javaName.startsWith("set") || javaName.startsWith("get"))) {
			javaName = lowercaseFirst(javaName.substring(3));
		}
		return javaName;
	}

}
