/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator;

import static java.nio.charset.StandardCharsets.UTF_8;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.generator.Result.Status;
import com.sap.cds.generator.util.GeneratedFile;
import com.sap.cds.generator.util.GeneratedFile.Consumer;
import com.sap.cds.generator.writer.ModelWriter;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.impl.reader.issuecollector.IssueCollector;
import com.sap.cds.reflect.impl.reader.issuecollector.IssueCollectorFactory;

public class Cds4jCodegen {

	private static final Logger logger = LoggerFactory.getLogger(Cds4jCodegen.class);
	private static final IssueCollector issueCollector = IssueCollectorFactory.getIssueCollector(Cds4jCodegen.class);
	private final Configuration configuration;

	public Cds4jCodegen(Configuration configuration) {
		this.configuration = configuration;
	}

	public Result generate(CsnSupplier csn, Consumer consumer) {
		Cds4jCodegenRunner helper = new Cds4jCodegenRunner(logger, issueCollector);
		return helper.generate(csn, model -> generate(model, consumer));
	}

	private Status generate(CdsModel cdsModel, Consumer consumer) throws IOException {
		if (Cds4jCodegenRunner.noRelevantIssues(configuration.getParserMode(), configuration.getGeneratorMode())) {
			ModelWriter visitor = new ModelWriter(consumer, configuration, cdsModel);
			cdsModel.accept(visitor);
			writeMetaInf(visitor, consumer);
			return Status.SUCCESS;
		}
		return Status.FAILURE;
	}

	private static void writeMetaInf(ModelWriter visitor, Consumer consumer) throws IOException {
		// don't create META-INF/cds4j-codegen/services.generated, if no service POJOs are generated
		if (!visitor.getGeneratedServices().isEmpty()) {
			StringBuilder builder = new StringBuilder();
			visitor.getGeneratedServices().forEach(fqnService -> builder.append(fqnService).append("\n"));
			consumer.accept(new GeneratedFile() {
				@Override
				public URI getUri() {
					return URI.create("META-INF/cds4j-codegen/services.generated");
				}

				@Override
				public InputStream getContent() {
					return new ByteArrayInputStream(builder.toString().getBytes(UTF_8));
				}

				/**
				 * @return {@code true} to indicate it has been written in to resources folder.
				 */
				@Override
				public boolean isResource() {
					return true;
				}
			});
		}
	}


}
