/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.NamesUtils.argumentName;
import static com.sap.cds.generator.util.NamesUtils.constantName;
import static com.sap.cds.generator.util.NamesUtils.getterName;
import static com.sap.cds.generator.util.NamesUtils.methodName;
import static com.sap.cds.generator.util.NamesUtils.setterName;
import static com.sap.cds.generator.util.NamesUtils.suffixedClassName;
import static com.sap.cds.generator.util.TypeUtils.addStaticCreateForKeys;
import static com.sap.cds.generator.util.TypeUtils.addStaticFactoryMethods;
import static com.sap.cds.generator.util.TypeUtils.getAnonymousElements;
import static com.sap.cds.generator.util.TypeUtils.getManagedToOneFks;
import static com.sap.cds.generator.util.TypeUtils.getReturnType;
import static com.sap.cds.generator.util.TypeUtils.isAnonymousAspect;
import static com.sap.cds.generator.util.TypeUtils.listOf;
import static com.sap.cds.generator.writer.SpecWriterUtil.addCdsNameAnnotation;
import static com.sap.cds.generator.writer.SpecWriterUtil.addCdsPathAnnotation;
import static com.sap.cds.generator.writer.SpecWriterUtil.addFkStaticField;
import static com.sap.cds.generator.writer.SpecWriterUtil.addStaticField;
import static com.sap.cds.generator.writer.SpecWriterUtil.cdsNameAnnotation;
import static com.sap.cds.generator.writer.SpecWriterUtil.setJavaDoc;
import static com.sap.cds.util.CdsModelUtils.managedToOne;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.MethodStyle;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.generator.util.TypeUtils;
import com.sap.cds.generator.writer.ModelWriter.Context;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsElementDefinition;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsEvent;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.reflect.impl.CdsEventBuilder.EventProxy;
import com.sap.cds.reflect.impl.DraftAdapter;
import com.sap.cds.util.CdsModelUtils;
import com.palantir.javapoet.ClassName;
import com.palantir.javapoet.MethodSpec;
import com.palantir.javapoet.ParameterSpec;
import com.palantir.javapoet.ParameterizedTypeName;
import com.palantir.javapoet.TypeName;
import com.palantir.javapoet.TypeSpec;
import com.palantir.javapoet.TypeSpec.Builder;
import com.palantir.javapoet.WildcardTypeName;

public class CreateConsumptionInterfaceVisitor implements CdsVisitor {
	private static final ParameterizedTypeName MAP_STR2OBJ = ParameterizedTypeName.get(Types.MAP,
			Types.STRING, WildcardTypeName.subtypeOf(Object.class));
	private final TypeSpec.Builder builder;
	private final Configuration cfg;
	private final ClassName className;
	private final Context context;
	private final Set<String> parentElements;

	CreateConsumptionInterfaceVisitor(Builder builder, ClassName className, Context context, Collection<CdsStructuredType> parents) {
		this.builder = builder;
		this.cfg = context.config();
		this.className = className;
		this.context = context;
		this.parentElements = parents.stream().flatMap(t -> t.elements()
			.filter(CreateConsumptionInterfaceVisitor::isAnonymousType)
			.map(CdsElementDefinition::getName))
			.collect(Collectors.toSet());
	}

	@Override
	public void visit(CdsEntity entity) {
		builder.addSuperinterface(Types.CDS_DATA);
		ClassName javaName = NamesUtils.className(cfg, entity);

		addRefMethod(entity);
		addStaticFactoryMethods(builder, javaName);

		Map<String, ParameterSpec> keyElements = new LinkedHashMap<>();
		entity.keyElements()
			.filter(k -> !TypeUtils.isIgnored(k))
			.forEach(key ->
				keyElements.put(
					constantName(key),
					ParameterSpec.builder(getSetterParam(key), argumentName(cfg, key)).build()));

		// In CSN the keys can be in arbitrary order and the signatures will change when order of elements changes
		// To avoid that, additional create method generated only if entity has single key
		if (keyElements.size() == 1) {
			addStaticCreateForKeys(builder, javaName, keyElements);
		}

		builder.addAnnotation(cdsNameAnnotation(entity.getQualifiedName(), "$S"));
	}

	@Override
	public void visit(CdsEvent event) {
		builder.addSuperinterface(Types.CDS_DATA);
		addStaticFactoryMethods(builder, className);

		builder.addAnnotation(cdsNameAnnotation(event.getQualifiedName(), "$S"));
	}

	@Override
	public void visit(CdsStructuredType struct) {
		builder.addSuperinterface(Types.CDS_DATA);
		addStaticFactoryMethods(builder, NamesUtils.className(cfg, struct));

		builder.addAnnotation(cdsNameAnnotation(struct.getQualifiedName(), "$S"));
	}

	@Override
	public void visit(CdsArrayedType type) {
		if (!type.getQualifiedName().isEmpty() && type.getItemsType().isStructured()
			&& (type.getItemsType().getQualifiedName().isEmpty() && !type.getItemsType().isSimple())) {

			ClassName itemsInterfaceName = className.nestedClass(NamesUtils.ITEM_TYPE_NAME);
			TypeSpec.Builder innerInterfaceBuilder = TypeSpec
				.interfaceBuilder(itemsInterfaceName).addModifiers(Modifier.PUBLIC, Modifier.STATIC).addSuperinterface(Types.CDS_DATA);
			type.getItemsType().as(CdsStructuredType.class).elements().forEach(e ->
				e.accept(new CreateConsumptionInterfaceVisitor(innerInterfaceBuilder, className.nestedClass(NamesUtils.ITEM_TYPE_NAME), context, Set.of())));
			addStaticFactoryMethods(builder, itemsInterfaceName, innerInterfaceBuilder);
			builder.addType(innerInterfaceBuilder.build());
		}
	}

	@Override
	public void visit(CdsElement element) {
		if (context.isTenantDiscriminator(element) || TypeUtils.isIgnored(element)) {
			return;
		}
		addStaticField(builder, element);

		// Anonymous elements that come from the parent should not generate own interface
		// inside the derived entity, but must use the type from the base class.
		if (isAnonymousType(element) && !parentElements.contains(element.getName())) {
			ClassName elementName = NamesUtils.className(cfg, className, element);
			TypeName returnType = getReturnType(className, element, cfg);

			TypeSpec.Builder innerInterfaceBuilder = TypeSpec.interfaceBuilder(elementName)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);
			innerInterfaceBuilder.addSuperinterface(Types.CDS_DATA);

			CdsType type = element.getType();
			Stream<CdsElement> elements;
			if (type.isStructured() && type.getQualifiedName().isEmpty()) {
				elements = type.as(CdsStructuredType.class).elements();
			} else if (type.isArrayed()) {
				elements = type.as(CdsArrayedType.class).getItemsType().as(CdsStructuredType.class).elements();
			} else {
				elements = getAnonymousElements(element);
			}
			elements.forEach(e -> e
				.accept(new CreateConsumptionInterfaceVisitor(innerInterfaceBuilder, elementName, context, Set.of())));
			addStaticFactoryMethods(builder, elementName, innerInterfaceBuilder);
			populateGetter(element, returnType, false);
			populateSetter(element, returnType, false);
			builder.addType(innerInterfaceBuilder.build());
		} else {
			getter(element);
			setter(element);
		}
		if (cfg.fkAccessors() && managedToOne(element.getType()) && !element.getName().equals(DraftAdapter.DRAFT_ADMINISTRATIVE_DATA)) {
			addFkStaticField(builder, element);
			addFkMethods(element);
		}

	}

	private void addRefMethod(CdsEntity entity) {
		if (!entity.isAbstract()) {
			TypeName returnType = suffixedClassName(context.config(), entity);
			MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("ref").returns(returnType)
					.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);

			builder.addMethod(methodBuilder.build());
		}
	}

	private void setter(CdsElement attribute) {
		TypeName paramType = getSetterParam(attribute);

		if (paramType == null) {
			return;
		}
		populateSetter(attribute, paramType, false);
	}

	private void populateSetter(CdsElement attribute, TypeName paramType, boolean cdsPath) {
		if (!(TypeUtils.isIgnored(attribute))) {
			String setter;
		    TypeName returnType;

			if (cfg.getMethodStyle() == MethodStyle.BEAN) {
				returnType = TypeName.VOID;
				setter = setterName(cfg, attribute);
			} else {
				returnType = className;
				setter = methodName(cfg, attribute);
			}

			MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(setter).returns(returnType)
					.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT).addParameter(paramType, argumentName(cfg, attribute));
			addJavaDoc(attribute, methodBuilder);
			if (cdsPath) {
				addCdsPathAnnotation(methodBuilder, attribute.getName(), setter);
			} else {
				addCdsNameAnnotation(cfg, methodBuilder, attribute);
			}
			builder.addMethod(methodBuilder.build());
		}
	}

	private TypeName getSetterParam(CdsElement element) {
		if (!cfg.getStrictSetters() && element.getType().isAssociation()) {
			if (CdsModelUtils.isSingleValued(element.getType())) {
				return MAP_STR2OBJ;
			}
			return listOf(WildcardTypeName.subtypeOf(MAP_STR2OBJ));
		}
		return getReturnType(className, element, cfg);
	}

	private void getter(CdsElement attribute) {
		TypeName returnType = getReturnType(className, attribute, cfg);

		if (returnType == null) {
			return;
		}
		populateGetter(attribute, returnType, false);
	}

	private void populateGetter(CdsElement attribute, TypeName returnType, boolean cdsPath) {
		if (!(TypeUtils.isIgnored(attribute))) {
			String getter;
			if (cfg.getMethodStyle().equals(MethodStyle.BEAN)) {
				getter = getterName(cfg, attribute);
			} else {
				getter = methodName(cfg, attribute);
			}
			MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(getter).returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);
			addJavaDoc(attribute, methodBuilder);
			if (cdsPath) {
				addCdsPathAnnotation(methodBuilder, attribute.getName(), getter);
			} else {
				addCdsNameAnnotation(cfg, methodBuilder, attribute);
			}
			builder.addMethod(methodBuilder.build());
		}
	}

	private void addJavaDoc(CdsElement attribute, MethodSpec.Builder methodBuilder) {
		if (!(attribute.getDeclaringType() instanceof EventProxy) && cfg.getDocs()) {
			setJavaDoc(attribute, methodBuilder);
		}
	}

	private void addFkMethods(CdsElement attribute) {
		// Add setter for managed to-one FK
		getManagedToOneFks(attribute).forEach(fkElement -> {
			if (context.isTenantDiscriminator(fkElement)) {
				return;
			}
			TypeName keyReturnType = getReturnType(className, fkElement, cfg);
			if (keyReturnType == null) {
				return;
			}
			populateSetter(fkElement, keyReturnType, true);
			populateGetter(fkElement, keyReturnType, true);
		});
	}

	private static boolean isAnonymousType(CdsElement element) {
		CdsType type = element.getType();
		if (type.isArrayed()) {
			if (type.getQualifiedName().startsWith(element.getDeclaringType().getQualifiedName())) {
				CdsType itemsType = type.as(CdsArrayedType.class).getItemsType();
				return itemsType.isStructured() && itemsType.getQualifiedName().isEmpty();
			} else {
				return false;
			}
		} else if (type.isStructured() && type.getQualifiedName().isEmpty()) {
			return true;
		} else return isAnonymousAspect(element);
	}
}
