/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.NamesUtils.normalizedConstantName;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.util.Set;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.util.TypeUtils;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsEnumType;
import com.sap.cds.reflect.CdsVisitor;
import com.palantir.javapoet.FieldSpec;
import com.palantir.javapoet.MethodSpec;
import com.palantir.javapoet.TypeName;
import com.palantir.javapoet.TypeSpec.Builder;

class CreateEnumConstantClassVisitor implements CdsVisitor {

	private final Builder result;

	private static final Set<Class<?>> WITH_PARSE = Set.of(LocalDate.class, Instant.class, LocalTime.class);
	private static final Set<Class<?>> WITH_VALUE_OF = Set.of(Long.class, Float.class, Double.class);
	private static final Set<Class<?>> SUPPORTED = Set.of(String.class, Boolean.class, Integer.class, Short.class, BigDecimal.class);

	CreateEnumConstantClassVisitor(Builder result) {
		this.result = result;
	}

	@Override
	public void visit(CdsEnumType<?> type) {
		if (isSupportedType(type.getType())) {
			result.addModifiers(Modifier.PUBLIC, Modifier.FINAL);
			result.addMethod(MethodSpec.constructorBuilder().addModifiers(Modifier.PRIVATE).build());

			TypeName itemType = TypeName.get(type.getJavaType());

			type.enumerals().entrySet().stream()
				.filter(e -> !TypeUtils.isIgnored(e.getValue()))
				.forEach(e -> {
					String constant = normalizedConstantName(e.getValue());
					FieldSpec.Builder field =
							FieldSpec.builder(itemType, constant, Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL);
						addInitialValue(field, type.getType(), e.getValue().value());
						result.addField(field.build());
					});
		}
	}

	private static boolean isSupportedType(CdsBaseType type) {
		return SUPPORTED.contains(type.javaType()) || WITH_VALUE_OF.contains(type.javaType()) || WITH_PARSE.contains(type.javaType());
	}

	private static void addInitialValue(FieldSpec.Builder field, CdsBaseType type, Object value) {
		if (type.javaType().equals(String.class)) {
			field.initializer("$S", value);
		} else if (type.javaType().equals(Boolean.class) || type.javaType().equals(Integer.class) || type.javaType().equals(Short.class)) {
			field.initializer("$L", value);
		} else if (type.javaType().equals(BigDecimal.class)) {
			field.initializer("new $T($S)", type.javaType(), value.toString());
		} else if (WITH_VALUE_OF.contains(type.javaType())) {
			field.initializer("$T.valueOf($S)", type.javaType(), value.toString());
		} else if (WITH_PARSE.contains(type.javaType())) {
			field.initializer("$T.parse($S)", type.javaType(), value.toString());
		} else {
			throw new UnsupportedOperationException("Enum type: " + type.javaType() + " is implemented yet.");
		}
	}
}
