/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.NamesUtils.constantName;
import static com.sap.cds.generator.util.NamesUtils.isValidTechnicalEntity;
import static com.sap.cds.generator.util.NamesUtils.qualifiedContextName;
import static com.sap.cds.generator.util.NamesUtils.suffixedClassName;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.lang.model.element.Modifier;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.ql.CdsName;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.palantir.javapoet.AnnotationSpec;
import com.palantir.javapoet.ClassName;
import com.palantir.javapoet.FieldSpec;
import com.palantir.javapoet.ParameterizedTypeName;
import com.palantir.javapoet.TypeName;
import com.palantir.javapoet.TypeSpec;

public class WrapperInterfaceCreator {

	private static final String DOT = ".";

	private final TypeSpec.Builder builder;
	private final CdsModel model;
	private final Configuration cfg;
	private final NamesUtils namesUtils;

	WrapperInterfaceCreator(TypeSpec.Builder builder, CdsModel model, Configuration cfg) {
		this.builder = builder;
		this.model = model;
		this.cfg = cfg;
		this.namesUtils = new NamesUtils(cfg);
	}

	public boolean generateInterface(String context) {
		List<CdsDefinition> definitionList = getEntities(context);
		definitionList.addAll(getEvents(context));
		definitionList.addAll(getActionsAndFunctions(context));
		if (definitionList.isEmpty()) {
			return false;
		}
		if (context != null) {
			builder.addAnnotation(AnnotationSpec.builder(CdsName.class).addMember("value", "$S", context).build());
			addStaticQualifiedAttribute(context);
		}

		definitionList.stream().forEach(d -> {
			if (d instanceof CdsEntity entity) {
				TypeName type;
				ClassName interfaceName = suffixedClassName(cfg, entity);
				type = ParameterizedTypeName.get(Types.CLASS, interfaceName);
				FieldSpec staticField = FieldSpec.builder(type, constantName(cfg, d))
						.addModifiers(Modifier.PUBLIC, Modifier.FINAL, Modifier.STATIC)
						.initializer("$T.class", interfaceName).build();
				builder.addField(staticField);
			}
		});
		return true;
	}

	private List<CdsDefinition> getEvents(String context) {
		if (context == null) {
			return model.events().filter(e -> !e.getQualifiedName().contains(DOT)).collect(Collectors.toList());
		}
		return model.events().filter(e -> filterDefinitions(context, e) && !namesUtils.isExcluded(e.getQualifiedName())).collect(Collectors.toList());
	}

	private List<CdsDefinition> getActionsAndFunctions(String context) {
		List<CdsDefinition> definitionList = new ArrayList<>();
		if (context == null) {
			model.actions().filter(a -> !a.getQualifiedName().contains(DOT)).forEach(definitionList::add);
			model.functions().filter(f -> !f.getQualifiedName().contains(DOT)).forEach(definitionList::add);
		}
		model.actions().filter(a -> filterDefinitions(context, a) && !namesUtils.isExcluded(a.getQualifiedName())).forEach(definitionList::add);
		model.functions().filter(f -> filterDefinitions(context, f) && !namesUtils.isExcluded(f.getQualifiedName())).forEach(definitionList::add);

		return definitionList;
	}

	private void addStaticQualifiedAttribute(String context) {
		FieldSpec staticEntityField = FieldSpec.builder(String.class, "CDS_NAME")
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL).initializer("$S", context).build();
		builder.addField(staticEntityField);
	}

	private List<CdsDefinition> getEntities(String context) {
		if (context == null) {
			return model.concreteEntities()
					.filter(e -> !e.getQualifiedName().contains(".")&& isValidTechnicalEntity(cfg, model, e))
					.collect(Collectors.toList());
		}
		return model.concreteEntities()
				.filter(e -> filterDefinitions(context, e) && !namesUtils.isExcluded(e.getQualifiedName()) && isValidTechnicalEntity(cfg, model, e))
				.collect(Collectors.toList());
	}

	private static boolean filterDefinitions(String context, CdsDefinition def) {
		String name = def.getName();
		String qualifiedName = def.getQualifiedName();
		return Objects.equals(context, qualifiedContextName(qualifiedName, name));
	}

}
