/************************************************************************
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.jdbc.spi;

import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import com.google.common.annotations.Beta;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnLock;
import com.sap.cds.ql.cqn.CqnLock.Mode;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSortSpecification;

@Beta
public interface StatementResolver {

	default Optional<String> collate(Locale locale) {
		return Optional.empty();
	}

	default Optional<String> collate(CqnSortSpecification o, Locale locale) {
		return collate(locale);
	}

	default Optional<String> statementWideCollation(CqnSelect select, Locale locale) {
		return Optional.empty();
	}

	default boolean supportsStatementWideCollation() {
		return false;
	}
	
	default String upsert(String table, Stream<String> keyColumns, Stream<String> upsertColumns,
			Stream<String> upsertValues) {
		throw new UnsupportedOperationException("Upsert is not supported on this data store");
	}

	default String lockMode(Mode mode) {
		return "FOR UPDATE";
	}

	default Optional<String> timeoutClause(int timeoutSeconds) {
		return Optional.empty();
	}

	default Stream<String> lockClause(CqnLock lock) {
		Stream.Builder<String> stream = Stream.builder();
		stream.add(lockMode(lock.mode()));
		lock.timeout().ifPresent(to -> timeoutClause(to).ifPresent(stream::add));

		return stream.build();
	}

	default String deleteAll(String tableName) {
		 return "DELETE FROM " + tableName;
	}

	default Optional<String> hints(Map<String, Object> hints) {
		return Optional.empty();
	}

	default CqnSelect preOptimize(CqnSelect select) {
		return select;
	}

	Select<?> applyTransformations(Select<?> select);
}
