package com.sap.cds.mtx.impl;

import com.sap.cds.CdsException;

import java.util.concurrent.TimeUnit;

/**
 * Value class that stores cache parameters for {@link  MetaDataAccessorImpl}
 */
public class CacheParams {
    private final long maximumSize;
    private final long expirationDuration;
    private final TimeUnit expirationDurationUnit;
    private final long refreshDuration;
    private final TimeUnit refreshDurationUnit;
    private final boolean synchronousRefresh;

    /**
     * @param maximumSize            maximum size of cache for csn and edmx model
     * @param expirationDuration     time in expiration duration unit after which model is deleted from cache after creation or last refresh
     * @param expirationDurationUnit time unit for expiration duration
     * @param refreshDuration        time in refresh duration unit after which model is reread from sidecar
     * @param refreshDurationUnit    time unit for refresh duration
     * @throws CdsException
     */
    public CacheParams(long maximumSize, long expirationDuration, TimeUnit expirationDurationUnit, long refreshDuration, TimeUnit refreshDurationUnit)
            throws CdsException {
        this(maximumSize, expirationDuration, expirationDurationUnit, refreshDuration, refreshDurationUnit, false);
    }

    /**
     * @param maximumSize            maximum size of cache for csn and edmx model
     * @param expirationDuration     time in expiration duration unit after which model is deleted from cache after creation or last refresh
     * @param expirationDurationUnit time unit for expiration duration
     * @param refreshDuration        time in refresh duration unit after which model is reread from sidecar
     * @param refreshDurationUnit    time unit for refresh duration
     * @param synchronousRefresh     guava cache is refreshed in the calling thread and not asynchronously
     * @throws CdsException
     */
    public CacheParams(long maximumSize, long expirationDuration, TimeUnit expirationDurationUnit, long refreshDuration, TimeUnit refreshDurationUnit,
                       boolean synchronousRefresh) throws CdsException {
        this.synchronousRefresh = synchronousRefresh;
        if (maximumSize < 0)
            throw new CdsException("Maximum cache size must be greater than or equal to 0");
        if (expirationDuration < 0)
            throw new CdsException("Expiration duration must be positive");
        if (expirationDurationUnit == null)
            throw new CdsException("Expiration duration unit must not be null");
        if (refreshDuration <= 0)
            throw new CdsException("Refresh duration must be greater than 0");
        if (refreshDurationUnit == null)
            throw new CdsException("Refresh duration unit must not be null");
        this.maximumSize = maximumSize;
        this.expirationDuration = expirationDuration;
        this.expirationDurationUnit = expirationDurationUnit;
        this.refreshDuration = refreshDuration;
        this.refreshDurationUnit = refreshDurationUnit;
    }

    /**
     * @return maximum cache size
     */
    public long getMaximumSize() {
        return maximumSize;
    }

    /**
     * @return time after which cache entry expires
     */
    public long getExpirationDuration() {
        return expirationDuration;
    }

    /**
     * @return time unit for time after which cache entry expires
     */
    public TimeUnit getExpirationDurationUnit() {
        return expirationDurationUnit;
    }

    /**
     * @return time after which model is reread
     */
    public long getRefreshDuration() {
        return refreshDuration;
    }

    /**
     * @return time unit for time after which model is reread
     */
    public TimeUnit getRefreshDurationUnit() {
        return refreshDurationUnit;
    }

    public boolean isSynchronousRefresh() {
        return synchronousRefresh;
    }
}
