/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.mtx.impl;

import com.google.common.base.Strings;
import com.google.common.base.Ticker;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.sap.cds.CdsException;
import com.sap.cds.mtx.MetaDataAccessor;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.impl.CdsModelReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.BiFunction;
import java.util.function.Consumer;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Class that provides access to CDS and Edmx models and caches them
 *
 * @param <M> Type used for the Edmx model
 */
public class MetaDataAccessorImpl<M> implements MetaDataAccessor<M> {
    private static final Logger logger = LoggerFactory.getLogger(MetaDataAccessorImpl.class);
    private final SidecarAccess sidecarAccess;
    private final LoadingCache<String, CacheEntry<CdsModel>> tenantToCdsModel;
    private final LoadingCache<EdmxCacheKey, CacheEntry<M>> tenantToEdmxModel;
    private final EdmxModelCreator<M> strToEdmx;
    //edmx, service name => Model
    private final BiFunction<String, String, M> strToEdmxFunction;
    private static volatile ExecutorService executorService;//NOSONAR

    /**
     * @param sidecarAccess Object of type {@link SidecarAccess}that provides access
     *                      to the node.js application sidecar/mtx via a rest API
     * @param cacheParams   Parameters that control size and lifecycle of cache for
     *                      cds and edmx models
     * @param strToEdmx     Function that converts an edmx model description given
     *                      as string into an edmx model
     * @param cacheTicker   Optional ticker used by guava cache for testing
     *                      purposes, use null for productive use
     */
    public MetaDataAccessorImpl(SidecarAccess sidecarAccess, CacheParams cacheParams, EdmxModelCreator<M> strToEdmx,
                                Ticker cacheTicker) {
        this(sidecarAccess, cacheParams, strToEdmx, null, cacheTicker);
    }

    /**
     * @param sidecarAccess     Object of type {@link SidecarAccess}that provides access
     *                          to the node.js application sidecar/mtx via a rest API
     * @param cacheParams       Parameters that control size and lifecycle of cache for
     *                          cds and edmx models
     * @param strToEdmxFunction BiFunction that converts an edmx model description given
     *                          as string into an edmx model.Parameters are edmx model as string
     *                          and service name
     * @param cacheTicker       Optional ticker used by guava cache for testing
     *                          purposes, use null for productive use
     */
    public MetaDataAccessorImpl(SidecarAccess sidecarAccess, CacheParams cacheParams,
                                Ticker cacheTicker,BiFunction<String, String, M> strToEdmxFunction) {
        this(sidecarAccess, cacheParams, null, strToEdmxFunction, cacheTicker);
    }

    private MetaDataAccessorImpl(SidecarAccess sidecarAccess, CacheParams cacheParams,
                                 EdmxModelCreator<M> strToEdmx,
                                 BiFunction<String, String, M> strToEdmxFunction,
                                 Ticker cacheTicker) {
        this.strToEdmx = strToEdmx;
        this.strToEdmxFunction = strToEdmxFunction;
        this.sidecarAccess = sidecarAccess;
        if (!cacheParams.isSynchronousRefresh() && executorService == null) {
            synchronized (MetaDataAccessorImpl.class) {
                if (executorService == null) {
                    executorService = Executors.newSingleThreadExecutor();
                }
            }
        }
        tenantToCdsModel = buildCache(cacheParams, cacheTicker, this::loadCdsModel, executorService);
        tenantToEdmxModel = buildCache(cacheParams, cacheTicker, this::loadEdmxModel, executorService);
    }

    private static <K, V> LoadingCache<K, V> buildCache(CacheParams params, Ticker cacheTicker,
                                                        BiFunction<K, V, V> loader, ExecutorService executorService) {
        CacheBuilder<Object, Object> builder = CacheBuilder.newBuilder().maximumSize(params.getMaximumSize())
                .expireAfterAccess(params.getExpirationDuration(), params.getExpirationDurationUnit())
                .refreshAfterWrite(params.getRefreshDuration(), params.getRefreshDurationUnit());
        if (cacheTicker != null) {
            builder.ticker(cacheTicker);
        }
        if (params.isSynchronousRefresh()) {
            return builder.build(getSynchronousLoader(loader));
        } else {
            return builder.build(getAsynchronousLoader(loader, executorService));
        }
    }

    /**
     * Returns an edmx model
     *
     * @param tenantId    tenant identifier, must not be null
     * @param serviceName service name, can be null
     * @param language    language, can be null
     * @return the edmx model for tenant, service name and language
     */
    @Override
    public M getEdmx(String tenantId, String serviceName, String language) throws CdsException {
        try {
            return tenantToEdmxModel.getUnchecked(new EdmxCacheKey(tenantId, serviceName, language)).getEntry();
        } catch (UncheckedExecutionException e) {
            throw new CdsException(e);
        }
    }

    /**
     * Returns a Cds model
     *
     * @param tenantId tenant id, mustn't be null
     * @return the {@link CdsModel} for the specified tenant
     */
    @Override
    public CdsModel getCdsModel(String tenantId) throws CdsException {
        try {
            return tenantToCdsModel.getUnchecked(tenantId).getEntry();
        } catch (UncheckedExecutionException e) {
            throw new CdsException(e);
        }
    }

    private CacheEntry<CdsModel> loadCdsModel(String tenantId, CacheEntry<CdsModel> oldEntry) {
        String eTag = oldEntry != null ? oldEntry.getETag() : null;
        ModelAndInformation csnModel = sidecarAccess.getCsn(tenantId, eTag);
        if (csnModel.isNotModified()) {
            return oldEntry;
        }
        return new CacheEntry<CdsModel>(CdsModelReader.read(csnModel.getModel(), true), csnModel.getETag());
    }

    private CacheEntry<M> loadEdmxModel(EdmxCacheKey key, CacheEntry<M> oldEntry) {
        String eTag = oldEntry != null ? oldEntry.getETag() : null;
        ModelAndInformation edmxModel = sidecarAccess.getEdmx(key.getTenantId(), key.getServiceName(),
                key.getLanguage(), eTag);
        if (edmxModel.isNotModified()) {
            return oldEntry;
        }
        if (strToEdmx != null) {
            return new CacheEntry<>(strToEdmx.parse(edmxModel.getModel(), key.getServiceName()), edmxModel.getETag());
        } else {
            return new CacheEntry<>(strToEdmxFunction.apply(edmxModel.getModel(), key.getServiceName()), edmxModel.getETag());
        }
    }


    @FunctionalInterface
    public static interface EdmxModelCreator<M> {
        M parse(String emdx, String serviceName);
    }

    @Override
    public void refresh(String tenantId) {
        cacheActions(tenantId, tenantToCdsModel::refresh, tenantToEdmxModel::refresh);
    }

    @Override
    public void evict(String tenantId) {
        cacheActions(tenantId, tenantToCdsModel::invalidate, tenantToEdmxModel::invalidate);
    }

    public void cacheActions(String tenantId, Consumer<String> cdsAction, Consumer<EdmxCacheKey> edmxAction) {
        Set<String> tenants = tenantToCdsModel.asMap().keySet();
        tenants.stream().filter(k -> k.equals(tenantId)).forEach(cdsAction);
        Set<EdmxCacheKey> keys = tenantToEdmxModel.asMap().keySet();
        keys.stream().filter(k -> k.tenantId.equals(tenantId)).forEach(edmxAction);
    }

    private static <K, V> CacheLoader<K, V> getAsynchronousLoader(BiFunction<K, V, V> loader,
                                                                  ExecutorService executorService) {
        return new CacheLoader<K, V>() {
            @Override
            public V load(K key) {
                return loader.apply(key, null);
            }

            @Override
            public ListenableFuture<V> reload(K key, V oldValue) {
                ListenableFutureTask<V> readModelTask = ListenableFutureTask.create(() -> {
                    try {
                        return loader.apply(key, oldValue);
                    } catch (Exception e) {// NOSONAR
                        logger.error("Error when model was reread: {}", e.getMessage());
                        return oldValue;
                    }
                });
                executorService.execute(readModelTask);
                return readModelTask;
            }
        };
    }

    private static <K, V> CacheLoader<K, V> getSynchronousLoader(BiFunction<K, V, V> loader) {
        return new CacheLoader<K, V>() {
            @Override
            public V load(K key) {
                return loader.apply(key, null);
            }

            @Override
            public ListenableFuture<V> reload(K key, V oldValue) throws Exception {
                checkNotNull(key);
                checkNotNull(oldValue);
                return Futures.immediateFuture(loader.apply(key, oldValue));
            }
        };
    }

    private static class CacheEntry<T> {
        private final T entry;
        private final String eTag;

        public CacheEntry(T entry, String eTag) {
            this.entry = entry;
            this.eTag = eTag.trim();
        }

        public T getEntry() {
            return entry;
        }

        public String getETag() {
            return eTag;
        }
    }

    private static class EdmxCacheKey {

        private final String tenantId;
        private final String serviceName;
        private final String language;

        public EdmxCacheKey(String tenantId, String serviceName, String language) throws CdsException {
            if (Strings.isNullOrEmpty(tenantId))
                throw new CdsException("Tenant id must not be null");
            this.tenantId = tenantId;
            this.serviceName = serviceName;
            this.language = language;
        }

        public String getTenantId() {
            return tenantId;
        }

        public String getServiceName() {
            return serviceName;
        }

        public String getLanguage() {
            return language;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o)
                return true;
            if (o == null || getClass() != o.getClass())
                return false;
            EdmxCacheKey key = (EdmxCacheKey) o;
            return tenantId.equals(key.tenantId) && Objects.equals(serviceName, key.serviceName)
                    && Objects.equals(language, key.language);
        }

        @Override
        public int hashCode() {
            return Objects.hash(tenantId, serviceName, language);
        }

    }
}
