/*******************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.mtx.impl;

import com.sap.cds.CdsException;
import com.sap.cds.mtx.MetaDataAccessor;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.impl.CdsModelReader;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Locale;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiFunction;

/**
 * Class that provides access to CDS and Edmx models read from java resource folders
 */
public class MetaDataAccessorSingleModelImpl<M> implements MetaDataAccessor<M> {
    private static final Logger logger = LoggerFactory.getLogger(MetaDataAccessorSingleModelImpl.class);
    private static final int BUFFER_SIZE = 1024;
    private final CdsModel cdsModel;
    private final String edmxFolder;
    private final ConcurrentHashMap<ServiceAndLanguage, M> keyToEdmx = new ConcurrentHashMap<ServiceAndLanguage, M>();
    //edmx as string,service name, model
    private final BiFunction<String, String, M> edmxstrAndServiceToEdmxModel;

    /**
     * @param cdsModelPath                 Path to the csn-file,including file name
     * @param edmxFolder                   Path of edmx folder, null and empty are allowed
     * @param edmxstrAndServiceToEdmxModel Function that converts edmx model read as string into generic type M
     * @throws IOException
     */
    public MetaDataAccessorSingleModelImpl(String cdsModelPath, String edmxFolder,
                                           BiFunction<String, String, M> edmxstrAndServiceToEdmxModel) throws IOException {
        if (cdsModelPath != null && !cdsModelPath.isEmpty()) {
            try (InputStream is = MetaDataAccessorSingleModelImpl.class.getClassLoader().getResourceAsStream(cdsModelPath)) {
                if (is == null) {
                    throw new IOException("Cds model " + cdsModelPath + " not found");
                }
                cdsModel = CdsModelReader.read(is);
            }
        } else {
            logger.warn("No model path provided");
            cdsModel = null;
        }
        this.edmxFolder = edmxFolder;
        this.edmxstrAndServiceToEdmxModel = edmxstrAndServiceToEdmxModel;
    }

    /**
     * @param tenantId    The tenant identifier, not considered by the code as model is the same for all tenants
     * @param serviceName Service name
     * @param language    Language, can be null
     * @return
     * @throws CdsException
     */
    @Override
    public M getEdmx(String tenantId, String serviceName, String language) throws CdsException {
        if (serviceName == null || serviceName.isEmpty()) {
            throw new CdsException("No service name specified");
        }
        if (language == null) {
            language = "";
        }
        ServiceAndLanguage key = new ServiceAndLanguage(language, serviceName);
        try {
            return keyToEdmx.computeIfAbsent(key, this::readEmdxFromResource);
        } catch (RuntimeException e) {
            throw new CdsException(e);
        }
    }

    private M readEmdxFromResource(ServiceAndLanguage key) throws RuntimeException {
        if (edmxstrAndServiceToEdmxModel == null) {
            throw new RuntimeException("No function to convert string into edmx model provided");
        }
        String edmxAsString;
        try (InputStream is = MetaDataAccessorSingleModelImpl.class.getClassLoader()
                .getResourceAsStream(getResourceName(key.service, key.language))) {
            if (is == null) {
                throw new IOException("Edmx file " + getResourceName(key.service, key.language) + " not found");
            }
            edmxAsString = IOUtils.toString(is, StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return edmxstrAndServiceToEdmxModel.apply(edmxAsString, key.service);
    }

    private String getResourceName(String serviceName, String language) {
        String completePathStr;
        if (edmxFolder == null || edmxFolder.isEmpty()) {
            completePathStr = serviceName;
        } else {
            completePathStr = edmxFolder + "/" + serviceName;
        }
        if (language != null && !language.isEmpty()) {
            completePathStr += "_" + language.toLowerCase(Locale.ENGLISH);
        }
        completePathStr += ".xml";
        return completePathStr;
    }

    /**
     * Returns a Cds model
     *
     * @param tenantId tenant id, mustn't be null
     * @return the {@link CdsModel} for the specified tenant
     */
    @Override
    public CdsModel getCdsModel(String tenantId) throws CdsException {
        return cdsModel;
    }

    @Override
    public void refresh(String tenantId) {
        //intentionally left empty, no cache
    }

    @Override
    public void evict(String tenantId) {
        //intentionally left empty, no cache
    }

    private static class ServiceAndLanguage {
        private final String language;
        private final String service;

        public ServiceAndLanguage(String language, String service) {
            this.language = language;
            this.service = service;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            ServiceAndLanguage that = (ServiceAndLanguage) o;
            return Objects.equals(language, that.language) &&
                    service.equals(that.service);
        }

        @Override
        public int hashCode() {
            return Objects.hash(language, service);
        }
    }
}
