package com.sap.cds.mtx.impl;

import java.io.IOException;
import java.text.MessageFormat;

import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;
import com.sap.cds.CdsCommunicationException;

/**
 * Class that is responsible for communication with node.js application
 * sidecar/mtx via its REST API
 */

public class SidecarAccess {
    private static final Logger logger = LoggerFactory.getLogger(SidecarAccess.class);
    private static final String CSN_PATH = "/mtx/v1/metadata/csn/";
    private static final String EDMX_PATH = "/mtx/v1/metadata/edmx/";
    private static final String APPLICATION_JSON = "application/json";
    private static final String AUTHENTIFICATION_SCHEME = "Bearer";

    private final String sidecarBaseUrl;
    private final ClientCredentialJwtAccess clientCredentialJwtAccess;

    /**
     * @param sidecarBaseUrl            URL of sidecar application without path
     *                                  specification
     * @param clientCredentialJwtAccess object that is responsible for JWT retrieval
     */
    public SidecarAccess(String sidecarBaseUrl, ClientCredentialJwtAccess clientCredentialJwtAccess) {
        this.sidecarBaseUrl = sidecarBaseUrl;
        this.clientCredentialJwtAccess = clientCredentialJwtAccess;
    }

    /**
     * Returns csn model as string, as returned by sidecar
     *
     * @param tenantId tenant identifier
     * @param eTag     entity tag
     * @return the {@link ModelAndInformation}
     * @throws CdsCommunicationException
     */
    public ModelAndInformation getCsn(String tenantId, String eTag) throws CdsCommunicationException {
        return callSidecar(clientCredentialJwtAccess.getJwt(), getCsnUrl(tenantId), eTag);
    }

    /**
     * Returns edmx model as string, as returned by sidecar
     *
     * @param tenantId    tenant identifier
     * @param serviceName service name
     * @param language    language
     * @param eTag        entity tag
     * @return the {@link ModelAndInformation}
     * @throws CdsCommunicationException
     */
    public ModelAndInformation getEdmx(String tenantId, String serviceName, String language, String eTag)
            throws CdsCommunicationException {
        return callSidecar(clientCredentialJwtAccess.getJwt(), getEdmxUrl(tenantId, serviceName, language), eTag);
    }

    private String getCsnUrl(String tenantId) {
        return sidecarBaseUrl + CSN_PATH + tenantId;
    }

    private String getEdmxUrl(String tenantId, String serviceName, String language) {
        String url = sidecarBaseUrl + EDMX_PATH + tenantId;
        if (!Strings.isNullOrEmpty(serviceName)) {
            url += "?name=" + serviceName;
            if (!Strings.isNullOrEmpty(language)) {
                url += "&language=" + language;
            }
        } else if (!Strings.isNullOrEmpty(language)) {
            url += "?language=" + language;
        }
        return url;
    }

    private ModelAndInformation callSidecar(String jwt, String url, String eTag) throws CdsCommunicationException {
        try (CloseableHttpClient client = HttpClientBuilder.create().build()) {
            logger.debug("Call sidecar on url {} ", url);
            HttpUriRequest request = RequestBuilder.get(url)
                    .setHeader(HttpHeaders.AUTHORIZATION, AUTHENTIFICATION_SCHEME + " " + jwt)
                    .setHeader(HttpHeaders.ACCEPT, APPLICATION_JSON).build();
            if (!Strings.isNullOrEmpty(eTag)) {
                request.setHeader(HttpHeaders.IF_NONE_MATCH, eTag);
            }
            try (CloseableHttpResponse response = client.execute(request)) {
                int httpStatusCode = response.getStatusLine().getStatusCode();
                switch (httpStatusCode) {
                    case HttpStatus.SC_NOT_MODIFIED:
                        logger.debug("Received not modified status");
                        return new ModelAndInformation(null, eTag, true);
                    case HttpStatus.SC_OK:
                        if (response.containsHeader(HttpHeaders.ETAG)) {
                            eTag = response.getLastHeader(HttpHeaders.ETAG).getValue();
                        }
                        String model = EntityUtils.toString(response.getEntity());
                        logger.debug("Received the following model from sidecar: {}", model);
                        return new ModelAndInformation(model, eTag, false);
                    default:
                        String contentAsString = EntityUtils.toString(response.getEntity());
                        String message = MessageFormat.format("Sidecar returned with status {0} and message {1}",
                                httpStatusCode, contentAsString);
                        logger.error(message);
                        throw new CdsCommunicationException(message, httpStatusCode);
                }
            } catch (IOException e) {
                throw new CdsCommunicationException("Communication error with sidecar.", e);
            }
        } catch (IOException e) {
            throw new CdsCommunicationException("Could not establish connection to sidecar.", e);
        }
    }

}
