/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.mtx.impl;

import com.sap.cds.CdsCommunicationException;
import com.sap.cloud.security.xsuaa.client.ClientCredentials;
import com.sap.cloud.security.xsuaa.client.DefaultOAuth2TokenService;
import com.sap.cloud.security.xsuaa.client.OAuth2TokenResponse;
import com.sap.cloud.security.xsuaa.client.XsuaaDefaultEndpoints;
import com.sap.cloud.security.xsuaa.tokenflows.TokenFlowException;
import com.sap.cloud.security.xsuaa.tokenflows.XsuaaTokenFlows;

import java.time.Instant;

/**
 * Class responsible for retrieval of a new JWT from XSUAA service via client
 * credential flow
 */
public class ClientCredentialJwtReader {
    private final XsuaaTokenFlows tokenFlows;

    /**
     * @param xsuaaParams parameters of xsuaa service instance
     */
    public ClientCredentialJwtReader(XsuaaParams xsuaaParams) {
        tokenFlows = new XsuaaTokenFlows(new DefaultOAuth2TokenService(),
                new XsuaaDefaultEndpoints(xsuaaParams.getUrl()),
                new ClientCredentials(xsuaaParams.getClientId(), xsuaaParams.getClientSecret()));
    }

    /**
     * Retrieve a new JWT
     *
     * @return the {@link OAuth2TokenResponse}
     * @throws CdsCommunicationException
     */
    public Response getJwt() throws CdsCommunicationException {
        try {
            OAuth2TokenResponse response = tokenFlows.clientCredentialsTokenFlow().execute();
            return new Response(response.getExpiredAt(), response.getAccessToken());
        } catch (TokenFlowException e) {
            throw new CdsCommunicationException("Could not retrieve JWT.", e);
        }
    }

    public static class Response {
        private final Instant expiredAt;
        private final String accessToken;

        public Response(Instant expiredAt, String accessToken) {
            this.expiredAt = expiredAt;
            this.accessToken = accessToken;
        }

        public boolean isValid() {
            return Instant.now().isBefore(expiredAt);
        }

        public String getAccessToken() {
            return accessToken;
        }

    }
}
