package com.sap.cds.mtx.impl;

import com.google.common.base.Strings;
import com.sap.cds.CdsCommunicationException;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultHttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.DestinationAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpClientAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.exception.DestinationAccessException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Class that is responsible for communication with node.js application
 * sidecar/mtx via its REST API
 */

public class SidecarAccess {
	private static final Logger logger = LoggerFactory.getLogger(SidecarAccess.class);
	private static final String CSN_PATH = "/mtx/v1/metadata/csn/";
	private static final String EDMX_PATH = "/mtx/v1/metadata/edmx/";
	private static final String APPLICATION_JSON = "application/json";
	private static final String AUTHENTIFICATION_SCHEME = "Bearer";
	private static final String MTX_SIDECAR_DESTINATION = "com.sap.cds.mtxSidecar";

	private final String sidecarBaseUrl;
	private final ClientCredentialJwtAccess clientCredentialJwtAccess;
	private final AtomicReference<HttpDestination> atomicSidecarHttpDestination = new AtomicReference<>();

	/**
	 * @param sidecarBaseUrl            URL of sidecar application without path
	 *                                  specification
	 * @param clientCredentialJwtAccess object that is responsible for JWT retrieval
	 */
	public SidecarAccess(String sidecarBaseUrl, ClientCredentialJwtAccess clientCredentialJwtAccess) {
		this.sidecarBaseUrl = sidecarBaseUrl;
		this.clientCredentialJwtAccess = clientCredentialJwtAccess;
	}

	/**
	 * Returns csn model as string, as returned by sidecar
	 *
	 * @param tenantId tenant identifier
	 * @param eTag     entity tag
	 * @return the {@link ModelAndInformation}
	 * @throws CdsCommunicationException
	 */
	public ModelAndInformation getCsn(String tenantId, String eTag) throws CdsCommunicationException {
		return callSidecar(clientCredentialJwtAccess.getJwt(), getCsnUrl(tenantId), eTag);
	}

	/**
	 * Returns edmx model as string, as returned by sidecar
	 *
	 * @param tenantId    tenant identifier
	 * @param serviceName service name
	 * @param language    language
	 * @param eTag        entity tag
	 * @return the {@link ModelAndInformation}
	 * @throws CdsCommunicationException
	 */
	public ModelAndInformation getEdmx(String tenantId, String serviceName, String language, String eTag)
			throws CdsCommunicationException {
		return callSidecar(clientCredentialJwtAccess.getJwt(), getEdmxUrl(tenantId, serviceName, language), eTag);
	}

	private String getCsnUrl(String tenantId) {
		return CSN_PATH + tenantId;
	}

	private String getEdmxUrl(String tenantId, String serviceName, String language) {
		String url = EDMX_PATH + tenantId;
		if (!Strings.isNullOrEmpty(serviceName)) {
			url += "?name=" + serviceName;
			if (!Strings.isNullOrEmpty(language)) {
				url += "&language=" + language;
			}
		} else if (!Strings.isNullOrEmpty(language)) {
			url += "?language=" + language;
		}
		return url;
	}

	private ModelAndInformation callSidecar(String jwt, String url, String eTag) throws CdsCommunicationException {
		HttpClient client = getHttpClient();
		logger.debug("Call sidecar on url {} ", url);
		HttpUriRequest request = RequestBuilder.get(url)
				.setHeader(HttpHeaders.AUTHORIZATION, AUTHENTIFICATION_SCHEME + " " + jwt)
				.setHeader(HttpHeaders.ACCEPT, APPLICATION_JSON).build();
		if (!Strings.isNullOrEmpty(eTag)) {
			request.setHeader(HttpHeaders.IF_NONE_MATCH, eTag);
		}
		try (CloseableHttpResponse response = (CloseableHttpResponse) client.execute(request)) {
			int httpStatusCode = response.getStatusLine().getStatusCode();
			switch (httpStatusCode) {
				case HttpStatus.SC_NOT_MODIFIED:
					logger.debug("Received not modified status");
					return new ModelAndInformation(null, eTag, true);
				case HttpStatus.SC_OK:
					if (response.containsHeader(HttpHeaders.ETAG)) {
						eTag = response.getLastHeader(HttpHeaders.ETAG).getValue();
					}
					String model = EntityUtils.toString(response.getEntity());
					logger.debug("Received the following model from sidecar: {}", model);
					return new ModelAndInformation(model, eTag, false);
				default:
					String contentAsString = EntityUtils.toString(response.getEntity());
					String message = MessageFormat.format("Sidecar returned with status {0} and message {1}",
							httpStatusCode, contentAsString);
					logger.error(message);
					throw new CdsCommunicationException(message, httpStatusCode);
			}
		} catch(IOException e) {
			throw new CdsCommunicationException("Communication error with sidecar.", e);
		}
	}

	private HttpDestination getSidecarDestination() throws CdsCommunicationException {
		HttpDestination sidecarHttpDestination = atomicSidecarHttpDestination.get();
		if (sidecarHttpDestination == null) {
			try {
				sidecarHttpDestination = DestinationAccessor.getDestination(MTX_SIDECAR_DESTINATION).asHttp();
			} catch(DestinationAccessException e) {
				try {
					sidecarHttpDestination = DefaultHttpDestination.builder(sidecarBaseUrl).name(MTX_SIDECAR_DESTINATION).build();
				} catch(IllegalArgumentException illegalArgumentException) {
					throw new CdsCommunicationException(illegalArgumentException);
				}
			}
			if (!atomicSidecarHttpDestination.compareAndSet(null, sidecarHttpDestination)) {
				return atomicSidecarHttpDestination.get();
			}
		}
		return sidecarHttpDestination;
	}

	private HttpClient getHttpClient() throws CdsCommunicationException {
		return HttpClientAccessor.getHttpClient(getSidecarDestination());
	}
}
