/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.mtx.impl;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiFunction;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Ticker;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.sap.cds.CdsException;
import com.sap.cds.mtx.MetaDataAccessor;
import com.sap.cds.mtx.ModelId;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.impl.CdsModelReader;

/**
 * Class that provides access to CDS and Edmx models and caches them
 *
 * @param <M> Type used for the Edmx model
 */
public class MetaDataAccessorImpl<M> implements MetaDataAccessor<M> {
	private static final long NANOS_TO_SECONDS = 1000 * 1000 * 1000l;
	private static final Logger logger = LoggerFactory.getLogger(MetaDataAccessorImpl.class);
	private final SidecarAccess sidecarAccess;
	private final Cache<CdsModel> modelIdToCdsModel;
	private final Cache<M> modelIdToEdmxModel;
	private final EdmxModelCreator<M> strToEdmx;
	private final ExecutorService executorService;

	/**
	 * @param sidecarAccess Object of type {@link SidecarAccess}that provides access
	 *                      to the node.js application sidecar/mtx via a rest API
	 * @param cacheParams   Parameters that control size and lifecycle of cache for
	 *                      cds and edmx models
	 * @param strToEdmx     Function that converts an edmx model description given
	 *                      as string into an edmx model
	 * @param cacheTicker   Optional ticker used by guava cache for testing
	 *                      purposes, use null for productive use
	 */
	public MetaDataAccessorImpl(SidecarAccess sidecarAccess, CacheParams cacheParams, EdmxModelCreator<M> strToEdmx,
			Ticker cacheTicker) {
		this.strToEdmx = strToEdmx;
		this.sidecarAccess = sidecarAccess;
		if (cacheTicker == null) {
			cacheTicker = Ticker.systemTicker();
		}

		if (cacheParams.isSynchronousRefresh()) {
			executorService = MoreExecutors.newDirectExecutorService();
		} else {
			executorService = Executors.newSingleThreadExecutor();
		}
		modelIdToCdsModel = new CdsCache(cacheParams, cacheTicker);
		modelIdToEdmxModel = new EdmxCache(cacheParams, cacheTicker);
	}

	@Override
	public CdsModel getCdsModel(ModelId key, int maxAgeSeconds) {
		return modelIdToCdsModel.getOrLoadIfStale(key, maxAgeSeconds);
	}

	@Override
	public M getEdmx(ModelId key, int maxAgeSeconds) throws CdsException {
		return modelIdToEdmxModel.getOrLoadIfStale(key, maxAgeSeconds);
	}

	@FunctionalInterface
	public static interface EdmxModelCreator<M> {
		M parse(String emdx, String serviceName);
	}

	@Override
	public void evict(String tenantId) {
		modelIdToCdsModel.evict(tenantId);
		modelIdToEdmxModel.evict(tenantId);
		// inform listeners
	}

	@Override
	public void refresh(String tenantId, int maxAgeSeconds) {
		modelIdToCdsModel.refresh(tenantId, maxAgeSeconds);
		modelIdToEdmxModel.refresh(tenantId, maxAgeSeconds);
		// inform listeners
	}

	private static <K, V> CacheLoader<K, V> getLoader(BiFunction<K, V, V> loader, ExecutorService executorService) {
		return new CacheLoader<K, V>() {
			@Override
			public V load(K key) {
				return loader.apply(key, null);
			}

			@Override
			public ListenableFuture<V> reload(K key, V oldValue) {
				ListenableFutureTask<V> readModelTask = ListenableFutureTask.create(() -> {
					try {
						return loader.apply(key, oldValue);
					} catch (Exception e) {// NOSONAR
						logger.error("Error when model was reread: {}", e);
						return oldValue;
					}
				});
				executorService.execute(readModelTask);
				return readModelTask;
			}
		};
	}

	private abstract class Cache<V> {
		private final Ticker ticker;
		private final LoadingCache<ModelId, Entry> guavaCache;

		protected Cache(CacheParams params, Ticker ticker) {
			CacheBuilder<Object, Object> builder = CacheBuilder.newBuilder().maximumSize(params.getMaximumSize())
					.expireAfterAccess(params.getExpirationDuration(), params.getExpirationDurationUnit())
					.refreshAfterWrite(params.getRefreshDuration(), params.getRefreshDurationUnit());
			if (ticker != null) {
				builder.ticker(ticker);
			}

			this.guavaCache = builder.build(getLoader(this::load, executorService));
			this.ticker = ticker;
		}

		public void evict(String tenantId) {
			forTenant(tenantId, guavaCache::invalidate);
		}

		public void refresh(String tenantId, int maxAgeSeconds) {
			forTenant(tenantId, k -> getOrLoadIfStale(k, maxAgeSeconds));
		}

		private void forTenant(String tenantId, Consumer<ModelId> action) {
			guavaCache.asMap().keySet().stream().filter(k -> k.getTenantId().equals(tenantId)).forEach(action);
		}

		public Entry getUnchecked(ModelId key) {
			return guavaCache.getUnchecked(key);
		}

		private void put(ModelId key, Entry entry) {
			guavaCache.put(key, entry);
		}

		public V getOrLoadIfStale(ModelId key, int maxAgeSeconds) {
			long maxAgeNanos = maxAgeSeconds * NANOS_TO_SECONDS;
			long now = ticker.read();
			Entry entry;
			try {
				entry = getUnchecked(key);
			} catch (UncheckedExecutionException e) {
				throw new CdsException(e);
			}
			if ((now - entry.refreshed()) > maxAgeNanos) {
				// sync load
				Entry loaded = load(key, entry);
				if (loaded != entry) {
					put(key, loaded);
					entry = loaded;
				}
			}

			return entry.getEntry();
		}

		private Entry load(ModelId key, Entry oldEntry) {
			String eTag = oldEntry != null ? oldEntry.getETag() : null;
			ModelAndInformation model = access(key, eTag);
			if (oldEntry != null && model.isNotModified()) {
				oldEntry.refresh();
				return oldEntry;
			}

			// model has changed -> notify
			return new Entry(parse(key, model.getModel()), model.getETag());
		}

		abstract ModelAndInformation access(ModelId key, String eTag);

		abstract V parse(ModelId key, String model);

		private class Entry {
			private final V entry;
			private final String eTag;
			private final AtomicLong refreshed = new AtomicLong(ticker.read());

			public Entry(V entry, String eTag) {
				this.entry = entry;
				this.eTag = eTag.trim();
			}

			public V getEntry() {
				return entry;
			}

			public String getETag() {
				return eTag;
			}

			public void refresh() {
				refreshed.set(ticker.read());
			}

			public long refreshed() {
				return refreshed.get();
			}

		}

	}

	private class EdmxCache extends Cache<M> {

		public EdmxCache(CacheParams params, Ticker ticker) {
			super(params, ticker);
		}

		@Override
		ModelAndInformation access(ModelId key, String eTag) {
			return sidecarAccess.getEdmx(key, eTag);
		}

		@Override
		M parse(ModelId key, String model) {
			return strToEdmx.parse(model, key.getServiceName().orElse(null));
		}

	}

	private class CdsCache extends Cache<CdsModel> {

		public CdsCache(CacheParams params, Ticker ticker) {
			super(params, ticker);
		}

		@Override
		ModelAndInformation access(ModelId key, String eTag) {
			return sidecarAccess.getCsn(key, eTag);
		}

		@Override
		CdsModel parse(ModelId key, String csn) {
			return CdsModelReader.read(csn, true);
		}

	}

}
