package com.sap.cds.mtx.impl;

import org.apache.http.client.methods.RequestBuilder;

import com.sap.cds.CdsCommunicationException;
import com.sap.cds.mtx.ModelId;

/**
 * Class that is responsible for communication with node.js application
 * sidecar/mtx via its REST API
 */

public class SidecarAccessV1 extends AbstractSidecarAccess implements SidecarAccess {
	private static final String CSN_PATH = "/mtx/v1/metadata/csn/";
	private static final String EDMX_PATH = "/mtx/v1/metadata/edmx/";

	/**
	 * @param sidecarBaseUrl URL of sidecar application without path specification
	 * @param authenticator  object that is responsible for authentication
	 * 
	 */
	public SidecarAccessV1(String sidecarBaseUrl, Authenticator authenticator) {
		super(sidecarBaseUrl, authenticator);
	}

	/**
	 * Returns csn model as string, as returned by sidecar
	 *
	 * @param id   the model identifier
	 * @param eTag entity tag
	 * @return the {@link ModelAndInformation}
	 * @throws CdsCommunicationException
	 */
	public ModelAndInformation getCsn(ModelId id, String eTag) throws CdsCommunicationException {
		String url = getCsnUrl(id);
		RequestBuilder builder = RequestBuilder.get(url);
		return callSidecar(builder.build(), eTag);
	}

	/**
	 * Returns edmx model as string, as returned by sidecar
	 *
	 * @param id   the model identifier
	 * @param eTag entity tag
	 * @return the {@link ModelAndInformation}
	 * @throws CdsCommunicationException
	 */
	public ModelAndInformation getEdmx(ModelId id, String eTag) throws CdsCommunicationException {
		String url = getEdmxUrl(id);
		RequestBuilder builder = RequestBuilder.get(url);
		return callSidecar(builder.build(), eTag);
	}

	private String getCsnUrl(ModelId id) {
		return CSN_PATH + id.getTenantId();
	}

	private String getEdmxUrl(ModelId id) {
		UrlBuilder url = new UrlBuilder(EDMX_PATH + id.getTenantId());
		id.getServiceName().ifPresent(s -> url.param("name", s));
		id.getLanguage().ifPresent(l -> url.param("language", l));

		return url.build();
	}

	private static class UrlBuilder {

		private StringBuilder url;
		private char c = '?';

		public UrlBuilder(String path) {
			url = new StringBuilder(path);
		}

		public void param(String name, String value) {
			url.append(c);
			url.append(name);
			url.append("=");
			url.append(value); // TODO encode
			c = '&';
		}

		public String build() {
			return url.toString();
		}

	}
}
