/*******************************************************************
 * © 2021 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.mtx.impl;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.concurrent.atomic.AtomicReference;

import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;
import com.sap.cds.CdsCommunicationException;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultHttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.DestinationAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpClientAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.exception.DestinationAccessException;

public class AbstractSidecarAccess {

	private static final Logger logger = LoggerFactory.getLogger(AbstractSidecarAccess.class);
	private static final String MTX_SIDECAR_DESTINATION = "com.sap.cds.mtxSidecar";
	private static final String APPLICATION_JSON = "application/json";

	private final String sidecarBaseUrl;
	private final Authenticator authenticator;
	private final AtomicReference<HttpDestination> atomicSidecarHttpDestination = new AtomicReference<>();

	/**
	 * @param sidecarBaseUrl
	 * @param authenticator
	 */
	public AbstractSidecarAccess(String sidecarBaseUrl, Authenticator authenticator) {
		this.sidecarBaseUrl = sidecarBaseUrl;
		this.authenticator = authenticator;
	}

	private HttpDestination getSidecarDestination() throws CdsCommunicationException {
		HttpDestination sidecarHttpDestination = atomicSidecarHttpDestination.get();
		if (sidecarHttpDestination == null) {
			try {
				sidecarHttpDestination = DestinationAccessor.getDestination(MTX_SIDECAR_DESTINATION).asHttp();
			} catch (DestinationAccessException e) {
				try {
					sidecarHttpDestination = DefaultHttpDestination.builder(sidecarBaseUrl)
							.name(MTX_SIDECAR_DESTINATION).build();
				} catch (IllegalArgumentException illegalArgumentException) {
					throw new CdsCommunicationException(illegalArgumentException);
				}
			}
			if (!atomicSidecarHttpDestination.compareAndSet(null, sidecarHttpDestination)) {
				return atomicSidecarHttpDestination.get();
			}
		}
		return sidecarHttpDestination;
	}

	private ModelAndInformation executeRequest(HttpUriRequest request, String eTag) {
		logger.debug("Call sidecar on url {} ", request.getURI());

		HttpClient client = HttpClientAccessor.getHttpClient(getSidecarDestination());
		try (CloseableHttpResponse response = (CloseableHttpResponse) client.execute(request)) {
			int httpStatusCode = response.getStatusLine().getStatusCode();
			switch (httpStatusCode) {
			case HttpStatus.SC_NOT_MODIFIED:
				logger.debug("Received not modified status");
				return new ModelAndInformation(null, eTag, true);
			case HttpStatus.SC_OK:
				if (response.containsHeader(HttpHeaders.ETAG)) {
					eTag = response.getLastHeader(HttpHeaders.ETAG).getValue();
				}
				String model = EntityUtils.toString(response.getEntity());
				logger.debug("Received a [modified] model from sidecar");
				return new ModelAndInformation(model, eTag, false);
			default:
				String contentAsString = EntityUtils.toString(response.getEntity());
				String message = MessageFormat.format("Sidecar returned with status {0} and message {1}",
						httpStatusCode, contentAsString);
				throw new CdsCommunicationException(message, httpStatusCode);
			}
		} catch (IOException e) {
			throw new CdsCommunicationException("Communication error with sidecar.", e);
		}
	}

	protected ModelAndInformation callSidecar(HttpUriRequest request, String eTag) {
		authenticator.getAuthorization().ifPresent(auth -> request.setHeader(HttpHeaders.AUTHORIZATION, auth));
		request.setHeader(HttpHeaders.ACCEPT, APPLICATION_JSON);
		if (!Strings.isNullOrEmpty(eTag)) {
			request.setHeader(HttpHeaders.IF_NONE_MATCH, eTag);
		}
		return executeRequest(request, eTag);
	}

}
