package com.sap.cds.mtx.impl;

import static org.apache.http.HttpHeaders.CONTENT_TYPE;

import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import org.apache.http.HttpEntity;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.StringEntity;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.CdsCommunicationException;
import com.sap.cds.mtx.ModelId;

/**
 * Class that is responsible for communication with node.js application
 * sidecar/mtx via its REST API
 */

public class SidecarAccessV2 extends AbstractSidecarAccess implements SidecarAccess {
	/**
	 * 
	 */
	private static final String BASE_PATH = "/mtx/v2/model/";
	private static final String CSN_PATH = BASE_PATH + "csn/";
	private static final String EDMX_PATH = BASE_PATH + "edmx/";
	private static final String APPLICATION_JSON = "application/json";
	private static final ObjectMapper mapper = new ObjectMapper();

	/**
	 * @param sidecarBaseUrl URL of sidecar application without path specification
	 * @param authenticator  object that is responsible for authentication
	 */
	public SidecarAccessV2(String sidecarBaseUrl, Authenticator authenticator) {
		super(sidecarBaseUrl, authenticator);
	}

	/**
	 * Returns csn model as string, as returned by sidecar
	 *
	 * @param id   the model identifier
	 * @param eTag entity tag
	 * @return the {@link ModelAndInformation}
	 * @throws CdsCommunicationException
	 */
	public ModelAndInformation getCsn(ModelId id, String eTag) throws CdsCommunicationException {
		RequestBuilder builder = RequestBuilder.post(CSN_PATH).setHeader(CONTENT_TYPE, APPLICATION_JSON)
				.setEntity(entity(getCsnBody(id)));
		return callSidecar(builder.build(), eTag);
	}

	/**
	 * Returns edmx model as string, as returned by sidecar
	 *
	 * @param id   the model identifier
	 * @param eTag entity tag
	 * @return the {@link ModelAndInformation}
	 * @throws CdsCommunicationException
	 */
	public ModelAndInformation getEdmx(ModelId id, String eTag) throws CdsCommunicationException {
		RequestBuilder builder = RequestBuilder.post(EDMX_PATH).setHeader(CONTENT_TYPE, APPLICATION_JSON)
				.setEntity(entity(getEdmxBody(id)));
		return callSidecar(builder.build(), eTag);
	}

	private static Map<String, Object> getCsnBody(ModelId id) {
		Map<String, Object> body = new HashMap<>();
		body.put("tenant", id.getTenantId());
		body.put("toggles", id.getFeatures());
		return body;
	}

	private static Map<String, Object> getEdmxBody(ModelId id) {
		Map<String, Object> body = new HashMap<>();
		body.put("tenant", id.getTenantId());
		body.put("toggles", id.getFeatures());
		id.getServiceName().ifPresent(s -> body.put("service", s));
		id.getLanguage().ifPresent(l -> body.put("locale", l));

		return body;
	}

	private static HttpEntity entity(Map<String, Object> object) {
		try {
			String json = mapper.writerWithDefaultPrettyPrinter().writeValueAsString(object);
			return new StringEntity(json);
		} catch (UnsupportedEncodingException e) {
			throw new CdsCommunicationException("unsupported encoding", e);
		} catch (JsonProcessingException e) {
			throw new CdsCommunicationException("malformed model id", e);
		}
	}

}
