/*******************************************************************
 * © 2021 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.impl;

import static com.sap.cds.util.CdsModelUtils.isCascading;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;

import com.sap.cds.impl.parser.token.RefSegmentImpl;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.util.CdsModelUtils.CascadeType;

public class DeleteCascader {
	private final CdsEntity entity;
	private final Set<String> visited = new HashSet<>();
	private final LinkedList<CqnDelete> deletes = new LinkedList<>();
	private CqnStructuredTypeRef ref;
	private CqnPredicate filter;

	private DeleteCascader(CdsEntity entity) {
		this.entity = entity;
	}

	public static DeleteCascader create(CdsEntity entity) {
		return new DeleteCascader(entity);
	}

	public DeleteCascader from(CqnStructuredTypeRef ref) {
		this.ref = ref;
		return this;
	}

	public DeleteCascader where(Optional<CqnPredicate> filter) {
		filter.ifPresent(f -> f.accept(new CqnVisitor() {
			@Override
			public void visit(CqnExistsSubquery query) {
				throw new UnsupportedOperationException("Cascading delete is not supported for where exists");
			}
		}));
		this.filter = filter.orElse(null);
		return this;
	}

	public void cascade(Consumer<CqnDelete> action) {
		StructuredType<?> path = CQL.to(RefSegmentImpl.copy(ref.segments()));
		if (filter != null) {
			path.filter(ref.targetSegment().filter().map(f -> CQL.and(f, filter)).orElse((Predicate) filter));
		}
		cascade(path, entity);
		deletes.forEach(d -> action.accept(d));
	}

	private void cascade(StructuredType<?> path, CdsEntity entity) {
		entity.associations().filter(a -> isCascading(CascadeType.DELETE, a)) //
				.forEach(assoc -> cascade(path, assoc));
	}

	private void cascade(StructuredType<?> path, CdsElement association) {
		CdsAssociationType assocType = association.getType();
		CdsEntity target = assocType.getTarget();
		if (!visited.add(association.getQualifiedName())) {
			throw new UnsupportedOperationException("Cascading delete is not supported for cyclic models");
		}
		StructuredType<?> p = path.to(association.getName());
		deletes.addFirst(Delete.from(p));
		cascade(p, target);
	}
}
