/************************************************************************
 * © 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.impl;

import static com.sap.cds.DataStoreConfiguration.MAX_BATCH_SIZE;
import static com.sap.cds.DataStoreConfiguration.MAX_BATCH_SIZE_DEFAULT;
import static com.sap.cds.ResultBuilder.selectedRows;
import static com.sap.cds.impl.ContextImpl.context;
import static com.sap.cds.impl.ExceptionHandler.chainNextExceptions;
import static com.sap.cds.impl.RowImpl.row;
import static com.sap.cds.impl.builder.model.StructuredTypeImpl.structuredType;
import static com.sap.cds.reflect.CdsBaseType.cdsType;
import static com.sap.cds.util.CdsModelUtils.element;
import static com.sap.cds.util.CdsModelUtils.entity;
import static com.sap.cds.util.CqnStatementUtils.$JSON;
import static com.sap.cds.util.CqnStatementUtils.containsRef;
import static com.sap.cds.util.CqnStatementUtils.isMediaType;
import static com.sap.cds.util.CqnStatementUtils.toManyExpands;
import static com.sap.cds.util.DataUtils.resolvePathAndAdd;
import static java.lang.System.arraycopy;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toList;

import java.io.Reader;
import java.lang.reflect.UndeclaredThrowableException;
import java.sql.BatchUpdateException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Joiner;
import com.google.common.collect.Sets;
import com.sap.cds.CdsDataStore;
import com.sap.cds.CdsDataStoreConnector.Capabilities;
import com.sap.cds.CdsDataStoreException;
import com.sap.cds.CdsException;
import com.sap.cds.CdsLockTimeoutException;
import com.sap.cds.Result;
import com.sap.cds.ResultBuilder;
import com.sap.cds.SessionContext;
import com.sap.cds.impl.PreparedCqnStmt.Parameter;
import com.sap.cds.impl.builder.model.ElementRefImpl;
import com.sap.cds.impl.builder.model.ExpandBuilder;
import com.sap.cds.impl.builder.model.SelectList;
import com.sap.cds.impl.localized.LocaleUtils;
import com.sap.cds.impl.parser.JsonParser;
import com.sap.cds.impl.parser.StructDataParser;
import com.sap.cds.impl.parser.token.Jsonizer;
import com.sap.cds.impl.sql.SQLStatementBuilder;
import com.sap.cds.impl.sql.SQLStatementBuilder.SQLStatement;
import com.sap.cds.jdbc.spi.DbContext;
import com.sap.cds.jdbc.spi.ExceptionAnalyzer;
import com.sap.cds.jdbc.spi.ValueBinder;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.CdsDataException;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnLiteral;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.sap.cds.transaction.TransactionManager;
import com.sap.cds.transaction.TransactionRequiredException;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.CqnStatementUtils;
import com.sap.cds.util.DataUtils;
import com.sap.cds.util.OnConditionAnalyzer;
import com.sap.cds.util.PathExpressionResolver;

public class JDBCClient implements ConnectedClient {
	private static final Logger logger = LoggerFactory.getLogger(JDBCClient.class);
	private static final TimingLogger timed = new TimingLogger(logger);

	private final Supplier<SQLDataSourceAdapter> adapter;
	private final TransactionManager transactionManager;
	private final Supplier<Connection> ds;
	private final ValueBinder binder;
	private final ExceptionAnalyzer exceptionAnalyzer;
	private final Capabilities capabilities;

	private Context context;
	private int maxBatchSize;

	public JDBCClient(Context context, Supplier<Connection> ds, TransactionManager transactionManager) {
		this.context = context;
		this.ds = ds;
		this.transactionManager = transactionManager;
		DbContext dbContext = context.getDbContext();
		this.binder = dbContext.getBinder();
		this.adapter = () -> new JdbcDataSourceAdapter(this.context);
		this.exceptionAnalyzer = context.getDbContext().getExceptionAnalyzer();
		this.capabilities = context.getDbContext().getCapabilities();
		this.maxBatchSize = getMaxBatchSize(context);
	}

	@Override
	public PreparedCqnStatement prepare(CqnStatement statement) {
		if (statement.isSelect()) {
			return prepare(statement.asSelect());
		}
		SQLStatement stmt = adapter.get().process(statement);
		CdsEntity root = entity(context.getCdsModel(), statement.ref());
		CqnStructuredTypeRef ref = null;
		try {
			ref = statement.ref();
		} catch (CdsException ignore) {
			// ignore for now.
		}
		return PreparedCqnStmt.createUpdate(stmt.sql(), stmt.params(), ref, root);
	}

	public PreparedCqnStmt prepare(CqnSelect select) {
		CdsStructuredType targetType = CqnStatementUtils.targetType(context.getCdsModel(), select);
		CqnStructuredTypeRef ref = null;
		if (!CqnStatementUtils.containsPathExpression(select.where())) {
			ref = CqnStatementUtils.targetRef(select);
		}
		boolean optimizeToManyExpands = ref != null;
		extendSelectList(select, targetType, optimizeToManyExpands);
		SQLStatementBuilder.SQLStatement stmt = adapter.get().process(select);

		return PreparedCqnStmt.create(stmt.sql(), select.items(), select.excluding(), stmt.params(), ref, targetType);
	}

	private static void extendSelectList(CqnSelect select, CdsStructuredType targetType, boolean toManyMapping) {
		if (!select.isDistinct() && select.groupBy().isEmpty() && containsRef(select.items())) {
			// add key elements
			Set<String> elements = targetType.concreteNonAssociationElements().filter(CdsElement::isKey)
					.map(CdsElement::getName).collect(Collectors.toSet());
			if (toManyMapping) {
				// add mapping elements for to-many expand optimization
				toManyExpands(targetType, select.items()).filter(e -> pathExpand(targetType, e)).forEach(exp -> {
					CqnStructuredTypeRef ref = exp.ref();
					CdsElement assoc = element(targetType, ref.segments());
					try {
						Map<String, String> mapping = fkMapping(ref, assoc);
						((SelectList) exp).setElementMapping(mapping);
						elements.addAll(mapping.values());
						logger.debug("Expand to-many " + assoc.getQualifiedName() + " using path");
					} catch (Exception e) {
						logger.debug("Expand to-many " + assoc.getQualifiedName()
								+ " using parent-keys due to complex on condition", e);
					}
				});
			}
			CqnStatementUtils.selectHidden(elements, select);
		}
	}

	private static boolean pathExpand(CdsStructuredType parent, CqnExpand exp) {
		if (((ExpandBuilder<?>) exp).lazy() || exp.hasLimit()) {
			return false;
		}
		final String PATH = "path";
		CdsElement element = CdsModelUtils.element(parent, exp.ref().segments());
		String expandMethod = element.getAnnotationValue(CdsConstants.ANNOTATION_JAVA_EXPAND + ".using", PATH);
		return PATH.equals(expandMethod);
	}

	private static Map<String, String> fkMapping(CqnStructuredTypeRef ref, CdsElement toManyAssoc) {
		HashMap<String, String> mapping = new HashMap<>();
		new OnConditionAnalyzer(toManyAssoc, true).getFkMapping().forEach((k, val) -> {
			List<String> segments = ref.segments().stream().map(CqnReference.Segment::id).collect(toList());
			if (val.isRef() && !val.asRef().firstSegment().startsWith("$")) {
				segments.set(segments.size() - 1, val.asRef().lastSegment());
				mapping.put(k, Joiner.on('.').join(segments));
			}
		});
		return mapping;
	}

	private static int append(int[] arr, int[] elements, int pos) {
		arraycopy(elements, 0, arr, pos, elements.length);
		return pos + elements.length;
	}

	private static void rejectAutoCommit(Connection conn) throws SQLException {
		if (conn.getAutoCommit()) {
			throw new TransactionRequiredException("Connection must not be in auto-commit mode");
		}
	}

	@Override
	public void setContextVariable(String key, String value) {
		if (context.getDbContext().getCapabilities().supportsClientInfo()) {
			try (Connection conn = ds.get()) {
				conn.setClientInfo(key, value);
			} catch (SQLException e) {
				throw new CdsDataStoreException(String.format("Failed to set context variable '%s'", key), e);
			}
		}
	}

	@Override
	public ResultBuilder executeQuery(PreparedCqnStatement preparedStmt, Map<String, Object> parameterValues,
			CdsDataStore dataStore, boolean isTransactionRequired) {
		if (isTransactionRequired) {
			requireTransaction();
		}

		PreparedCqnStmt pcqn = (PreparedCqnStmt) preparedStmt;
		String sql = pcqn.toNative();

		CdsStructuredType targetType = pcqn.targetType();
		ResultBuilder result = timed.debug(() -> {
			try (Connection conn = ds.get(); PreparedStatement pstmt = conn.prepareStatement(sql)) {
				bindValues(pstmt, parameterValues, pcqn.parameters(), targetType);
				try (ResultSet rs = pstmt.executeQuery()) {
					return result(pcqn, parameterValues, rs, dataStore);
				}
			} catch (SQLException ex) {
				chainNextExceptions(ex);
				if (exceptionAnalyzer.isLockTimeout(ex)) {
					throw new CdsLockTimeoutException(targetType);
				}
				throw new CdsDataStoreException("Error executing the statement", ex);
			} catch (UndeclaredThrowableException ex) { // NOSONAR
				throw new CdsDataStoreException("Error executing the statement", ex);
			}
		}, "SQL >>{}<<", sql);
		logger.debug("SQL row count: {}", result.result().rowCount());

		return result;
	}

	@Override
	public int[] executeUpdate(PreparedCqnStatement preparedStmt, List<Map<String, Object>> parameterValues) {
		PreparedCqnStmt pcqn = (PreparedCqnStmt) preparedStmt;
		requireTransaction();

		String sql = pcqn.toNative();
		CdsEntity entity = (CdsEntity) pcqn.targetType();
		int[] rowCount = timed.debug(() -> {
			try (Connection conn = ds.get()) {
				rejectAutoCommit(conn);
				try (PreparedStatement pstmt = conn.prepareStatement(sql)) {
					List<Parameter> params = pcqn.parameters();
					if (parameterValues.size() > 1) {
						return executeBatch(pstmt, params, parameterValues, entity);
					}
					Map<String, Object> values = firstEntry(parameterValues);
					bindValues(pstmt, values, params, entity);
					return new int[] { pstmt.executeUpdate() };
				}
			} catch (CdsException e) {
				throw e;
			} catch (Exception e) {
				ExceptionHandler exHandler = new ExceptionHandler(entity, exceptionAnalyzer);
				throw exHandler.cdsException(firstEntry(parameterValues), e);
			}
		}, "SQL >>{}<<", sql);
		logger.debug("SQL affected rows: {}", Arrays.stream(rowCount).sum());

		return rowCount;
	}

	private int[] executeBatch(PreparedStatement pstmt, List<Parameter> params, List<Map<String, Object>> list,
			CdsEntity entity) throws SQLException {
		int row = 0;
		int rcPosition = 0;
		int[] result = new int[list.size()];
		try {
			for (Map<String, Object> entry : list) {
				bindValues(pstmt, entry, params, entity);
				pstmt.addBatch();
				row++;
				if (row % maxBatchSize == 0) {
					int[] rc = pstmt.executeBatch();
					rcPosition = append(result, rc, rcPosition);
				}
			}
			int[] rc = pstmt.executeBatch();
			rcPosition = append(result, rc, rcPosition); // NOSONAR
			return result;

		} catch (BatchUpdateException ex) {
			chainNextExceptions(ex);
			throw new ExceptionHandler(entity, exceptionAnalyzer).cdsBatchException(list, rcPosition, ex);
		}
	}

	private void requireTransaction() {
		if (!transactionManager.isActive()) {
			throw new TransactionRequiredException();
		}
	}

	private void bindValues(PreparedStatement pstmt, Map<String, Object> values, List<Parameter> accessors,
			CdsStructuredType entity) throws SQLException {
		for (int col = 1; col <= accessors.size(); col++) {
			Parameter param = accessors.get(col - 1);
			Object value = param.get(values);
			CdsBaseType type = param.type();
			if (value != null && Collection.class.isAssignableFrom(value.getClass())) {
				// TODO: docstore - refactor arrayed elements parameters to JSON parameter
				value = Jsonizer.json(value);
			}
			try {
				binder.setValue(pstmt, col, type, value);
			} catch (IllegalArgumentException | NullPointerException e) {
				throw new CdsDataException("Invalid value for '" + entity + "." + param.name() + "' of type " + type,
						e);
			}
		}
	}

	private ResultBuilder result(PreparedCqnStmt pcqn, Map<String, Object> parameterValues, ResultSet result,
			CdsDataStore dataStore) {

		Set<CqnExpand> expands = pcqn.selectListItems().stream().filter(CqnSelectListItem::isExpand)
				.map(CqnSelectListItem::asExpand).collect(Collectors.toSet());
		Set<CqnExpand> toManyExpands = expands.stream().filter(e -> !((SelectList) e).getElementMapping().isEmpty())
				.collect(Collectors.toSet());

		try {
			List<Map<String, Object>> rows = new ArrayList<>();
			boolean singleRow = handleRows(pcqn, result, dataStore, expands, toManyExpands, rows);
			if (!singleRow && !toManyExpands.isEmpty()) {
				handleExpands(pcqn, parameterValues, dataStore, toManyExpands, rows);
			}
			return selectedRows(rows);

		} catch (SQLException e) {
			chainNextExceptions(e);
			throw new CdsDataStoreException("Failed to process result set", e);
		}
	}

	private void handleExpands(PreparedCqnStmt pcqn, Map<String, Object> parameterValues, CdsDataStore dataStore,
			Set<CqnExpand> toManyExpands, List<Map<String, Object>> rows) {
		expand(pcqn.ref(), toManyExpands, rows, dataStore, parameterValues);
		if (!pcqn.excluding().isEmpty()) {
			rows.forEach(row -> row.keySet().removeAll(pcqn.excluding()));
		}
	}

	private boolean handleRows(PreparedCqnStmt pcqn, ResultSet result, CdsDataStore dataStore, Set<CqnExpand> expands,
			Set<CqnExpand> toManyExpands, List<Map<String, Object>> rows) throws SQLException {

		CdsStructuredType targetType = pcqn.targetType();
		Set<String> keys = CdsModelUtils.keyNames(targetType);
		int rowCounter = 0;
		List<CqnSelectListValue> slvs = pcqn.selectListItems().stream().flatMap(CqnSelectListItem::ofValue)
				.collect(toList());
		ResultSetMetaData meta = result.getMetaData();

		boolean singleRow = false;
		boolean hasNextRow = result.next();
		while (hasNextRow) {
			rowCounter++;

			AssociationLoader assocLoader = new AssociationLoader(dataStore, targetType);
			Map<String, Object> keyValues = new HashMap<>();
			keys.forEach(k -> keyValues.put(k, null));

			Map<String, Object> data = extractData(result, slvs, targetType, meta, assocLoader, keyValues);
			hasNextRow = result.next();
			if (rowCounter == 1 && !hasNextRow) {
				singleRow = true;
			}
			Set<CqnExpand> exp = singleRow ? expands : Sets.difference(expands, toManyExpands);
			exp.forEach(e -> assocLoader.expand(e, data));
			StructuredType<?> selfRef = ref(targetType, keyValues);
			if (singleRow || toManyExpands.isEmpty()) {
				pcqn.excluding().forEach(data::remove);
			}
			rows.add(row(data, selfRef));
		}
		return singleRow;
	}

	@SuppressWarnings("unchecked")
	private Map<String, Object> extractData(ResultSet result, List<CqnSelectListValue> slvs,
			CdsStructuredType targetType, ResultSetMetaData meta, AssociationLoader assocLoader,
			Map<String, Object> keyValues) throws SQLException {
		Map<String, Object> data = new HashMap<>();
		for (int i = 1; i <= meta.getColumnCount(); i++) {
			CqnSelectListValue slv = slvs.get(i - 1);
			Object value = getValue(result, i, targetType, slv);
			if (isSerializedJson(slv)) {
				mergeObject(keyValues, data, slv, (Map<String, Object>) value);
			} else {
				assocLoader.addValueOfRootEntity(slv, value);
				addValueTo(data, keyValues, slv, value);
			}
		}
		return data;
	}

	private void mergeObject(Map<String, Object> keyValues, Map<String, Object> data, CqnSelectListValue slv,
			Map<String, Object> mapValue) {
		CqnElementRef jsonRef = slv.asRef();
		String displayName = slv.displayName();
		String prefix = displayName.substring(0, displayName.lastIndexOf($JSON));

		mapValue.forEach((k, v) -> {
			ElementRef<?> innerRef = ElementRefImpl.elementRef(jsonRef);
			innerRef.targetSegment().id(k);
			CqnSelectListValue innerSlv = innerRef.as(prefix + k);
			addValueTo(data, keyValues, innerSlv, v);
		});
	}

	private boolean isSerializedJson(CqnSelectListValue slv) {
		// TODO docstore: nicer method for checking jsonish-type
		return slv.displayName().endsWith($JSON);
	}

	private void expand(CqnStructuredTypeRef ref, Collection<CqnExpand> expands, List<Map<String, Object>> rows,
			CdsDataStore dataStore, Map<String, Object> parameterValues) {
		if (ref != null && !expands.isEmpty()) {
			final String fkPrefix = "@fk:";
			expands.forEach(exp -> {
				List<CqnReference.Segment> segments = new ArrayList<>(ref.segments());
				segments.addAll(exp.ref().segments());
				List<CqnSelectListItem> expItems = new ArrayList<>(exp.items());
				Map<String, String> mapping = new HashMap<>();
				((SelectList) exp).getElementMapping().forEach((fk, v) -> {
					String alias = fkPrefix + fk;
					expItems.add(CQL.get(fk).as(alias));
					mapping.put(alias, "@" + v.replace('.', '_'));
				});

				CqnSelect sel = Select.from(CQL.to(segments)).columns(expItems).orderBy(exp.orderBy());
				// INNER JOIN leads to duplicates for many-to-many
				// -> transform ref path to where exists (semi-join)
				sel = PathExpressionResolver.resolvePath(context.getCdsModel(), sel);

				Result expResult = dataStore.execute(sel, parameterValues);
				DataUtils.merge(rows, expResult.list(), exp.displayName(), mapping, fkPrefix);
			});
		}
	}

	private Object getValue(ResultSet result, int i, CdsStructuredType targetType, CqnSelectListValue slv)
			throws SQLException {
		if (slv.value().isRef()) {
			CdsType type = element(targetType, slv.value().asRef()).getType();
			if (type.isArrayed()) {
				CdsType itemsType = type.as(CdsArrayedType.class).getItemsType();
				String json = binder.getValue(result, i, CdsBaseType.LARGE_STRING, false);
				return StructDataParser.parseArrayOf(itemsType, json);
			} else if (isSerializedJson(slv)) {
				Reader reader = binder.getValue(result, i, CdsBaseType.LARGE_STRING, true);
				return reader == null ? Collections.emptyMap() : JsonParser.map(reader);
			}
		}
		CdsBaseType cdsType = getCdsType(targetType, slv.value());
		return binder.getValue(result, i, cdsType, isMediaType(targetType, slv));
	}

	private void addValueTo(Map<String, Object> data, Map<String, Object> keys, CqnSelectListValue sli, Object value) {
		String displayName = sli.displayName();
		resolvePathAndAdd(data, displayName, value);
		if (sli.value().isRef()) {
			CqnElementRef ref = sli.value().asRef();
			String name = ref.firstSegment();
			if (ref.segments().size() == 1 && keys.containsKey(name)) {
				keys.put(name, value);
			}
		}
	}

	private Map<String, Object> firstEntry(List<Map<String, Object>> valueList) {
		return valueList.isEmpty() ? emptyMap() : valueList.get(0);
	}

	private static StructuredType<?> ref(CdsStructuredType type, Map<String, Object> keyValues) {
		if (type instanceof CdsEntity && !keyValues.containsValue(null)) {
			return structuredType(type.getQualifiedName()).matching(keyValues);
		}
		return null;
	}

	private static CdsBaseType getCdsType(CdsStructuredType rowType, CqnValue value) {
		Optional<String> type = value.type();
		if (type.isPresent()) {
			String cdsTypeName = type.get();
			try {
				return cdsType(cdsTypeName);
			} catch (CdsException e) {
				logger.warn("Failed to cast to {}", cdsTypeName);
				return null;
			}
		}
		if (value.isRef()) {
			CdsType t = element(rowType, value.asRef()).getType();
			if (t.isSimple()) {
				return t.as(CdsSimpleType.class).getType();
			}
		}
		if (value.isLiteral()) {
			CqnLiteral<?> literal = value.asLiteral();
			if (literal.isNumeric()) {
				return CdsBaseType.DECIMAL;
			}
			if (literal.isBoolean()) {
				return CdsBaseType.BOOLEAN;
			}
			if (literal.isString()) {
				return CdsBaseType.STRING;
			}
		}
		logger.debug("Cannot determine CDS type of {}", value);
		return null;
	}

	@Override
	public void setSessionContext(SessionContext session) {
		this.context = context(context.getCdsModel(), context.getDbContext(), session,
				context.getDataStoreConfiguration());
		this.maxBatchSize = getMaxBatchSize(context);
		this.setContextVariable("LOCALE", LocaleUtils.getLocaleString(session.getLocale()));
		this.setContextVariable("VALID-FROM",
				session.getValidFrom() != null ? session.getValidFrom().toString() : null);
		this.setContextVariable("VALID-TO", session.getValidTo() != null ? session.getValidTo().toString() : null);
		this.setContextVariable("APPLICATIONUSER",
				session.getUserContext().getId() != null ? session.getUserContext().getId() : null);
	}

	private static int getMaxBatchSize(Context context) {
		return Math.max(1, context.getDataStoreConfiguration().getProperty(MAX_BATCH_SIZE, MAX_BATCH_SIZE_DEFAULT));
	}

	@Override
	public Capabilities capabilities() {
		return capabilities;
	}

	@Override
	public void setRollbackOnly() {
		transactionManager.setRollbackOnly();
	}
}
