/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.impl;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.function.Supplier;

import javax.sql.DataSource;

import com.sap.cds.CdsCommunicationException;
import com.sap.cds.DataStoreConfiguration;
import com.sap.cds.connector.CdsDataStoreConnectorBuilder;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.transaction.TransactionManager;

public class JDBCDataStoreConnector extends AbstractDataStoreConnector {

	private final TransactionManager transactionManager;

	private JDBCDataStoreConnector(CdsModel cdsModel, Supplier<Connection> ds, TransactionManager transactionManager,
			DataStoreConfiguration dataStoreConfiguration) {
		super(cdsModel, ctx -> new JDBCClient(ctx, ds, transactionManager),
				DbContextFactoryRegister.context(dataStoreConfiguration, ds), dataStoreConfiguration);
		this.transactionManager = transactionManager;
	}

	/**
	 * @deprecated use the constructor with explicit DataStoreConfiguration
	 */
	@Deprecated
	public JDBCDataStoreConnector(CdsModel cdsModel, Supplier<Connection> ds, TransactionManager transactionManager) {
		this(cdsModel, ds, transactionManager, new SystemPropertyDataStoreConfiguration());
	}

	/**
	 * @deprecated use the constructor with explicit DataStoreConfiguration
	 */
	@Deprecated
	public JDBCDataStoreConnector(CdsModel cdsModel, DataSource ds, TransactionManager transactionManager) {
		this(cdsModel, () -> {
			try {
				return ds.getConnection();
			} catch (SQLException e) {
				throw new CdsCommunicationException("Cannot obtain connection to DB", e);
			}
		}, transactionManager);
	}

	public static Builder create(CdsModel model, TransactionManager transactionManager) {
		return new Builder(model, transactionManager);
	}

	public TransactionManager getTransactionManager() {
		return transactionManager;
	}

	public static class Builder implements CdsDataStoreConnectorBuilder {
		private final CdsModel model;
		private final TransactionManager transactionManager;
		private Supplier<Connection> connectionSupplier;
		private DataStoreConfiguration dataStoreConfiguration = new SystemPropertyDataStoreConfiguration();

		private Builder(CdsModel model, TransactionManager transactionManager) {
			this.model = model;
			this.transactionManager = transactionManager;
		}

		public CdsDataStoreConnectorBuilder datasource(DataSource datasource) {
			this.connectionSupplier = () -> {
				try {
					return datasource.getConnection();
				} catch (SQLException e) {
					throw new CdsCommunicationException("Cannot obtain connection to DB", e);
				}
			};
			return this;
		}

		public CdsDataStoreConnectorBuilder connection(Supplier<Connection> connectionSupplier) {
			this.connectionSupplier = connectionSupplier;
			return this;
		}

		public CdsDataStoreConnectorBuilder config(DataStoreConfiguration dataStoreConfiguration) {
			this.dataStoreConfiguration = dataStoreConfiguration;
			return this;
		}

		public JDBCDataStoreConnector build() {
			return new JDBCDataStoreConnector(model, connectionSupplier, transactionManager, dataStoreConfiguration);
		}
	}
}
