/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.impl.qat;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import com.sap.cds.impl.builder.model.ElementRefImpl;
import com.sap.cds.impl.sql.SqlMapping;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnValidationException;
import com.sap.cds.reflect.CdsElement;

public class Ref2QualifiedColumn implements Function<CqnElementRef, String> {

	private final QatSelectableNode root;
	private final QatSelectableNode outer;
	private final String rootName;

	public Ref2QualifiedColumn(Deque<QatSelectableNode> outer) {
		Iterator<QatSelectableNode> reverse = outer.descendingIterator();
		this.root = qatRoot(reverse.next());
		this.rootName = root.rowType().getQualifiedName();
		this.outer = reverse.hasNext() ? reverse.next() : null;
	}

	public String apply(CqnElementRef ref) {
		if (ref.firstSegment().equals(CqnExistsSubquery.OUTER)) {
			CqnElementRef subRef = ElementRefImpl.subRef(ref, 1);

			return qualifiedColumnName(subRef, outer);
		}

		return qualifiedColumnName(ref, root);
	}

	private String qualifiedColumnName(CqnElementRef ref, QatNode node) {
		boolean firstSegment = true;
		String tableAlias;
		String columnName;

		for (Segment s : ref.segments()) {
			if (firstSegment && s.id().equals(rootName)) {
				continue;
			}

			node = node.child(s.id(), s.filter());
			assertNotNull(ref, node);
			firstSegment = false;
		}

		if (node instanceof QatElementNode) {
			columnName = columnName(node);
		} else {
			throw new CqnValidationException(node.name() + " does not refer to an element");
		}
		do {
			node = node.parent();
		} while (node != null && !(node instanceof QatSelectableNode));
		assertNotNull(ref, node);

		tableAlias = ((QatSelectableNode) node).alias();

		return column(tableAlias, columnName);
	}

	private static String column(String alias, String col) {
		return alias + "." + col;
	}

	private static QatSelectableNode qatRoot(QatNode root) {
		Optional<QatNode> next;
		while ((next = childInSource(root)).isPresent()) {
			root = next.get();
		}

		return (QatSelectableNode) root;
	}

	private static Optional<QatNode> childInSource(QatNode root) {
		return root.children().stream().filter(QatNode::inSource).findFirst();
	}

	private String columnName(QatNode node) {
		QatElementNode elementNode = (QatElementNode) node;
		CdsElement cdsElement;
		if (node.parent() instanceof QatStructuredElementNode) {
			CdsElement element = elementNode.element();
			String columnName = SqlMapping.columnName(element);
			if (element.getName().equals(columnName)) {
				return structuredColumnName(node);
			}
			return columnName;
		} else {
			cdsElement = elementNode.element();
		}

		return SqlMapping.columnName(cdsElement);
	}

	private String structuredColumnName(QatNode node) {
		List<String> elementName = new ArrayList<>();
		elementName.add(((QatElementNode) node).element().getName());
		QatNode parent = node.parent();
		while (parent instanceof QatStructuredElementNode) {
			QatStructuredElementNode structNode = (QatStructuredElementNode) parent;
			elementName.add(structNode.element().getName());
			parent = structNode.parent();
		}
		Collections.reverse(elementName);
		return String.join("_", elementName);
	}

	private static void assertNotNull(CqnElementRef ref, QatNode node) {
		if (node == null) {
			throw new CqnValidationException("Unresolvable path expression: " + ref.toJson());
		}
	}

}
