/************************************************************************
 * © 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.impl.sql;

import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.sap.cds.impl.PreparedCqnStmt.DataParam;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsType;

public class SQLHelper {

	private static final char DQ = '\"';
	private static final char SQ = '\'';

	private SQLHelper() {
	}

	public static String commaSeparated(Stream<String> stream) {
		return stream.collect(Collectors.joining(", ", "(", ")"));
	}

	public static DataParam param(String path, CdsElement element) {
		CdsType type = element.getType();
		CdsBaseType baseType = type.isSimple() ? type.as(CdsSimpleType.class).getType() : null;
		return new DataParam(path, baseType);
	}

	public static String delimited(String undelimited) {
		return DQ + prefix(undelimited, DQ, DQ) + DQ;
	}

	public static String literal(String text) {
		return text == null ? "NULL" : SQ + prefix(text, SQ, SQ) + SQ;
	}

	public static String escapeLikePattern(char escapeChar, String text) {
		text = prefix(text, escapeChar, '_');
		text = prefix(text, escapeChar, '%');

		return text;
	}

	private static String prefix(String text, char escapeChar, char toBeEscaped) {
		for (int i = 0; i < text.length(); i++) {
			if (text.charAt(i) == toBeEscaped) {
				return prefixFrom(text, i, escapeChar, text.length(), toBeEscaped);
			}
		}

		return text;
	}

	private static String prefixFrom(String text, int from, char escapeChar, int length, char toBeEscaped) {
		char[] buffer = new char[length * 2];
		text.getChars(0, from, buffer, 0);
		int j = from;
		for (int i = from; i < length; i++) {
			char c = text.charAt(i);
			if (c == toBeEscaped) {
				buffer[j++] = escapeChar;
			}
			buffer[j++] = c;
		}

		return String.copyValueOf(buffer, 0, j);
	}

}
