/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.impl.sql;

import static com.sap.cds.impl.sql.SQLStatementBuilder.commaSeparated;
import static com.sap.cds.impl.sql.SpaceSeparatedCollector.joining;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import com.google.common.collect.Streams;
import com.sap.cds.SessionContext;
import com.sap.cds.impl.Context;
import com.sap.cds.impl.PreparedCqnStmt.Parameter;
import com.sap.cds.impl.builder.model.Conjunction;
import com.sap.cds.impl.qat.FromClauseBuilder;
import com.sap.cds.impl.qat.QatBuilder;
import com.sap.cds.impl.qat.QatSelectableNode;
import com.sap.cds.impl.qat.Ref2QualifiedColumn;
import com.sap.cds.jdbc.spi.StatementResolver;
import com.sap.cds.ql.cqn.CqnLock;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnSource;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.impl.SelectBuilder;

public class SelectStatementBuilder implements SQLStatementBuilder {

	private final Context context;
	private final CqnSelect select;
	private final Deque<QatSelectableNode> outer;
	private final StatementResolver statementResolver;
	private final SessionContext sessionContext;
	private final List<Parameter> params;

	private TokenToSQLTransformer toSQL;

	public SelectStatementBuilder(Context context, List<Parameter> params, CqnSelect select,
			Deque<QatSelectableNode> outer) {
		this.context = context;
		this.params = params;
		this.select = select;
		this.outer = append(outer, new QatBuilder(context, select, outer.size()).create());
		this.statementResolver = context.getDbContext().getStatementResolver();
		this.sessionContext = context.getSessionContext();
	}

	public SQLStatement build() {
		toSQL = new TokenToSQLTransformer(context, params, new Ref2QualifiedColumn(outer), outer);
		List<String> snippets = new ArrayList<>();
		with().forEach(snippets::add);
		snippets.add("SELECT");
		distinct().forEach(snippets::add);
		columns().forEach(snippets::add);
		from().forEach(snippets::add);
		where().forEach(snippets::add);
		groupBy().forEach(snippets::add);
		having().forEach(snippets::add);
		orderBy().forEach(snippets::add);
		limit().forEach(snippets::add);
		lock().forEach(snippets::add);
		String sql = snippets.stream().collect(joining());

		return new SQLStatement(sql, params);
	}

	private Stream<String> distinct() {
		if (select.isDistinct()) {
			return Stream.of("DISTINCT");
		}

		return Stream.empty();
	}

	private Stream<String> columns() {
		List<CqnSelectListItem> items = select.items();
		if (items.isEmpty()) {
			throw new IllegalStateException("select * not expected");
		}

		Stream.Builder<String> sql = Stream.builder();
		items.stream().flatMap(CqnSelectListItem::ofValue).forEach(slv -> {
			sql.add(",");
			sql.add(toSQL.apply(slv.value()));
			slv.alias().ifPresent(a -> {
				sql.add("as");
				sql.add(SQLHelper.delimited(a));
			});
		});

		return sql.build().skip(1);
	}

	private Stream<String> with() {
		FromClauseBuilder fromClauseBuilder = new FromClauseBuilder(context, params,
				((SelectBuilder<?>) select).hints());

		return fromClauseBuilder.with(outer);
	}

	private Stream<String> from() {
		FromClauseBuilder fromClauseBuilder = new FromClauseBuilder(context, params,
				((SelectBuilder<?>) select).hints());

		return Streams.concat(Stream.of("FROM"), fromClauseBuilder.sql(outer));
	}

	private Stream<String> where() {
		Optional<CqnPredicate> where = Conjunction.and(select.where(), select.search());
		CqnSource source = select.from();
		if (source.isRef()) {
			CqnStructuredTypeRef ref = source.asRef();
			where = Conjunction.and(where, ref.targetSegment().filter());
		}

		Stream.Builder<String> sql = Stream.builder();
		where.map(toSQL::toSQL).ifPresent(s -> {
			sql.add("WHERE");
			sql.add(s);
			Optional<String> collateClause = statementResolver.collateClause(sessionContext.getLocale());
			collateClause.ifPresent(cc -> sql.add(cc));
		});

		return sql.build();
	}

	private Stream<String> groupBy() {
		List<CqnValue> groupByTokens = select.groupBy();
		if (groupByTokens.isEmpty()) {
			return Stream.empty();
		}
		return Streams.concat(Stream.of("GROUP BY"), commaSeparated(groupByTokens.stream(), toSQL::apply));
	}

	private Stream<String> having() {
		Optional<CqnPredicate> having = select.having();

		Stream.Builder<String> sql = Stream.builder();
		having.map(toSQL::toSQL).ifPresent(s -> {
			sql.add("HAVING");
			sql.add(s);
			Optional<String> collateClause = statementResolver.collateClause(sessionContext.getLocale());
			collateClause.ifPresent(cc -> sql.add(cc));
		});

		return sql.build();
	}

	private Stream<String> orderBy() {
		List<CqnSortSpecification> orderBy = select.orderBy();
		if (orderBy.isEmpty()) {
			return Stream.empty();
		}
		return Streams.concat(Stream.of("ORDER BY"), commaSeparated(orderBy.stream(), this::sort));
	}

	private String sort(CqnSortSpecification o) {
		StringBuilder sort = new StringBuilder(toSQL.apply(o.value()));
		statementResolver.collateClause(sessionContext.getLocale()).ifPresent(cc -> sort.append(" " + cc));
		switch (o.order()) {
		case DESC:
			sort.append(" DESC NULLS LAST");
			break;
		case DESC_NULLS_FIRST:
			sort.append(" DESC NULLS FIRST");
			break;
		case ASC_NULLS_LAST:
			sort.append(" NULLS LAST");
			break;
		default: // ASC
			sort.append(" NULLS FIRST");
		}
		return sort.toString();
	}

	private Stream<String> limit() {
		Stream.Builder<String> sql = Stream.builder();
		select.limit().ifPresent(l -> {
			sql.add("LIMIT");
			l.limit().tokens().map(toSQL).forEach(sql::add);
			l.offset().ifPresent(o -> {
				sql.add("OFFSET");
				o.tokens().map(toSQL).forEach(sql::add);
			});
		});

		return sql.build();
	}

	private Stream<String> lock() {
		Stream.Builder<String> stream = Stream.builder();
		Optional<CqnLock> lock = select.getLock();
		lock.ifPresent(l -> {
			statementResolver.lockClause(l.mode()).ifPresent(stream::add);
			l.timeout().ifPresent(to -> {
				if (to > 0) {
					statementResolver.timeoutClause(to).ifPresent(stream::add);
				} else {
					stream.add("NOWAIT");
				}
			});
		});
		return stream.build();
	}

	private static Deque<QatSelectableNode> append(Deque<QatSelectableNode> prefix, QatSelectableNode newRoot) {
		ArrayDeque<QatSelectableNode> roots = new ArrayDeque<>(prefix);
		roots.add(newRoot);

		return roots;
	}

}
