/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.impl.sql;

import static com.sap.cds.impl.builder.model.ElementRefImpl.parse;
import static com.sap.cds.impl.localized.LocaleUtils.localizedEntityName;
import static com.sap.cds.impl.sql.SQLHelper.delimited;

import java.util.Locale;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.sap.cds.impl.docstore.DocStoreUtils;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.reflect.CdsAnnotation;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;

public class SqlMapping {

	private static final String CDS_PERSISTENCE_NAME = "cds.persistence.name";
	private final CdsStructuredType rowType;

	public SqlMapping(CdsStructuredType rowType) {
		this.rowType = rowType;
	}

	public String tableName() {
		CdsEntity entity = asEntity();

		return annotatedName().map(SQLHelper::delimited).orElseGet(() -> plainTableName(entity.getQualifiedName()));
	}

	public String localizedViewName() {
		CdsEntity entity = asEntity();

		return annotatedName().filter(n -> !uppercaseOnly(n)).map(n -> delimited(localizedEntityName(n)))
				.orElseGet(() -> plainTableName(localizedEntityName(entity)));
	}

	private boolean uppercaseOnly(String n) {
		return n.equals(n.toUpperCase(Locale.US));
	}

	private CdsEntity asEntity() {
		if (!(rowType instanceof CdsEntity)) {
			throw new IllegalStateException(rowType.getQualifiedName() + "is no entity");
		}

		return (CdsEntity) rowType;
	}

	private Optional<String> annotatedName() {
		return rowType.<String>findAnnotation(CDS_PERSISTENCE_NAME).map(CdsAnnotation::getValue);
	}

	public static String plainTableName(String qualifiedEntityName) {
		return qualifiedEntityName.replace(".", "_");
	}

	public String columnName(String elementName) {
		return rowType.findElement(elementName).map(SqlMapping::columnName)
				.orElseGet(() -> underscoreSeparated(parse(elementName)));
	}

	public String columnName(CqnElementRef ref) {
		return columnName(underscoreSeparated(ref));
	}

	public String columnName(Stream<? extends Segment> segments) {
		return columnName(underscoreSeparated(segments));
	}

	public static String columnName(CdsElement element) {

		Optional<CdsAnnotation<String>> persistenceName = element.findAnnotation(CDS_PERSISTENCE_NAME);
		if (persistenceName.isPresent()) {
			return delimited(persistenceName.get().getValue());
		}

		String name = element.getName();
		CdsStructuredType declaringType = element.getDeclaringType();

		if (DocStoreUtils.targetsDocStore(declaringType)) {
			return delimited(name);
		}
		return name;
	}

	private static String underscoreSeparated(CqnElementRef ref) {
		return underscoreSeparated(ref.segments().stream());
	}

	private static String underscoreSeparated(Stream<? extends Segment> segments) {
		return segments.map(Segment::id).collect(Collectors.joining("_"));
	}

	public CdsStructuredType getRowType() {
		return rowType;
	}

	public String jsonObjectPath(CqnElementRef ref) {
		return ref.segments().stream().map(seg -> "\"" + seg.id() + "\"").collect(Collectors.joining("."));
	}

}
