/************************************************************************
 * © 2021-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.jdbc.generic;

import static com.sap.cds.util.CdsTypeUtils.dateTime;
import static com.sap.cds.util.CdsTypeUtils.timestamp;

import java.io.InputStream;
import java.io.Reader;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZonedDateTime;
import java.util.Calendar;
import java.util.TimeZone;

import com.sap.cds.jdbc.spi.ValueBinder;
import com.sap.cds.reflect.CdsBaseType;

public abstract class AbstractValueBinder implements ValueBinder {

	protected AbstractValueBinder(int timestampFractionalSeconds) {
		this.timestampFractionalSeconds = timestampFractionalSeconds;
	}

	private static final ThreadLocal<Calendar> LOCAL_CALENDAR = new ThreadLocal<>(); // NOSONAR
	protected static final ThreadLocal<Calendar> UTC = // NOSONAR
			ThreadLocal.withInitial(() -> Calendar.getInstance(TimeZone.getTimeZone("UTC")));

	private final int timestampFractionalSeconds;

	@Override
	@SuppressWarnings("unchecked")
	public <T> T getValue(ResultSet result, int i, CdsBaseType cdsType, boolean isMediaType) throws SQLException {
		if (cdsType != null) {
			return (T) getValueFromResult(result, cdsType, isMediaType, i);
		}
		return (T) result.getObject(i);
	}

	@Override
	public void setValue(PreparedStatement pstmt, int i, CdsBaseType cdsType, Object value) throws SQLException {
		if (cdsType != null && value instanceof String) {
			String val = (String) value;
			switch (cdsType) {
			case TIMESTAMP:
				setInstant(pstmt, i, timestamp(val, timestampFractionalSeconds));
				break;
			case DATETIME:
				setInstant(pstmt, i, dateTime(val));
				break;
			case DATE:
				setLocalDate(pstmt, i, LocalDate.parse(val));
				break;
			case TIME:
				setLocalTime(pstmt, i, LocalTime.parse(val));
				break;
			default:
				pstmt.setObject(i, val);
			}
		} else if (value instanceof Instant) {
			setInstant(pstmt, i, (Instant) value);
		} else if (value instanceof LocalDate) {
			setLocalDate(pstmt, i, (LocalDate) value);
		} else if (value instanceof LocalTime) {
			setLocalTime(pstmt, i, (LocalTime) value);
		} else if (value instanceof ZonedDateTime) {
			setZonedDateTime(pstmt, i, (ZonedDateTime) value);
		} else if (value instanceof Timestamp) {
			setInstant(pstmt, i, ((Timestamp) value).toInstant());
		} else if (value instanceof java.sql.Time) {
			setLocalTime(pstmt, i, ((java.sql.Time) value).toLocalTime());
		} else if (value instanceof java.sql.Date) {
			setLocalDate(pstmt, i, ((java.sql.Date) value).toLocalDate());
		} else if (value instanceof byte[]) {
			pstmt.setBytes(i, (byte[]) value);
		} else if (value instanceof Reader) {
			setLargeString(pstmt, i, (Reader) value);
		} else if (value instanceof InputStream) {
			setLargeBinary(pstmt, i, (InputStream) value);
		} else {
			pstmt.setObject(i, value);
		}
	}

	protected abstract void setLocalTime(PreparedStatement pstmt, int i, LocalTime localTime) throws SQLException;

	protected abstract void setLocalDate(PreparedStatement pstmt, int i, LocalDate localDate) throws SQLException;

	protected abstract void setInstant(PreparedStatement pstmt, int i, Instant instant) throws SQLException;

	protected abstract void setLargeString(PreparedStatement result, int i, Reader reader) throws SQLException;

	protected abstract void setLargeBinary(PreparedStatement result, int i, InputStream stream) throws SQLException;

	protected abstract LocalTime getLocalTime(ResultSet result, int i) throws SQLException;

	protected abstract LocalDate getLocalDate(ResultSet result, int i) throws SQLException;

	protected abstract Instant getInstant(ResultSet result, int i) throws SQLException;

	protected abstract Reader getLargeString(ResultSet result, int i) throws SQLException;

	protected abstract InputStream getLargeBinary(ResultSet result, int i) throws SQLException;

	@SuppressWarnings("deprecation")
	private Object getValueFromResult(ResultSet result, CdsBaseType cdsType, boolean isMediaType, int i)
			throws SQLException {
		switch (cdsType) {
		case BOOLEAN:
			boolean bool = result.getBoolean(i);
			return result.wasNull() ? null : bool;
		case DATE:
			return getLocalDate(result, i);
		case DATETIME:
			return dateTime(getInstant(result, i));
		case DECIMAL:
		case HANA_SMALLDECIMAL:
		case DECIMAL_FLOAT:
			return result.getBigDecimal(i);
		case DOUBLE:
			double dbl = result.getDouble(i);
			return result.wasNull() ? null : dbl;
		case INTEGER:
			int in = result.getInt(i);
			return result.wasNull() ? null : in;
		case INTEGER64:
			long int64 = result.getLong(i);
			return result.wasNull() ? null : int64;
		case LARGE_BINARY:
			if (isMediaType) {
				return getLargeBinary(result, i);
			}
		case HANA_BINARY:
		case BINARY:
			return result.getBytes(i);
		case HANA_CLOB:
		case LARGE_STRING:
			if (isMediaType) {
				return getLargeString(result, i);
			}
		case STRING:
			String str = result.getString(i);
			return result.wasNull() ? null : str;
		case TIME:
			return getLocalTime(result, i);
		case TIMESTAMP:
			return timestamp(getInstant(result, i), timestampFractionalSeconds);
		case HANA_TINYINT:
		case HANA_SMALLINT:
			short smallInt = result.getShort(i);
			return result.wasNull() ? null : smallInt;
		case HANA_REAL:
			float real = result.getFloat(i);
			return result.wasNull() ? null : real;
		default:
			return result.getObject(i);
		}
	}

	private void setZonedDateTime(PreparedStatement pstmt, int i, ZonedDateTime zonedDateTime) throws SQLException {
		setInstant(pstmt, i, dateTime(zonedDateTime.toInstant()));
	}

	protected static Calendar getCalendarForDefaultTimeZone() {
		Calendar calendar = LOCAL_CALENDAR.get();
		if (null == calendar
				|| !calendar.getTimeZone().getDisplayName().equals(TimeZone.getDefault().getDisplayName())) {
			LOCAL_CALENDAR.set(Calendar.getInstance(TimeZone.getDefault()));
		}
		return LOCAL_CALENDAR.get();
	}

}
