/*******************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.jdbc.hana;

import java.util.Locale;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.CdsDataStoreConnector;
import com.sap.cds.CdsDataStoreConnector.Capabilities;
import com.sap.cds.DataStoreConfiguration;
import com.sap.cds.jdbc.generic.GenericDbContext;
import com.sap.cds.jdbc.hana.search.HanaSearchResolver;
import com.sap.cds.jdbc.spi.ExceptionAnalyzer;
import com.sap.cds.jdbc.spi.SearchResolver;
import com.sap.cds.jdbc.spi.StatementResolver;
import com.sap.cds.jdbc.spi.TableNameResolver;
import com.sap.cds.jdbc.spi.ValueBinder;
import com.sap.cds.reflect.CdsModel;

public final class HanaDbContext extends GenericDbContext {
	private static final Logger logger = LoggerFactory.getLogger(HanaDbContext.class);
	private final int majorVersion;

	public HanaDbContext(DataStoreConfiguration dataStoreConfiguration, int majorVersion) {
		super(dataStoreConfiguration);
		this.majorVersion = majorVersion;
	}

	@Override
	public TableNameResolver createTableNameResolver(Locale locale) {
		return new HanaTableNameResolver(dataStoreConfiguration, locale);
	}

	@Override
	public SearchResolver createSearchResolver(CdsModel model, Locale locale) {
		boolean useGenericSearchResolver = dataStoreConfiguration
				.getProperty(DataStoreConfiguration.USE_LOCALIZED_VIEW_SEARCH_RESOLVER, false);

		if (useGenericSearchResolver) {
			logger.debug("Switched to LocalizedViewSearchResolver due to provided configuration.");
			return new LocalizedViewSearchResolver(model, locale);
		}
		return new HanaSearchResolver(model, locale);
	}

	@Override
	public StatementResolver getStatementResolver() {
		return new HanaStatementResolver(dataStoreConfiguration, majorVersion);
	}

	@Override
	public ExceptionAnalyzer getExceptionAnalyzer() {
		return new HanaExceptionAnalyzer();
	}

	@Override
	public Capabilities getCapabilities() {
		return CdsDataStoreConnector.Capabilities.ALL;
	}

	@Override
	public ValueBinder getBinder() {
		return new HanaBinder();
	}
}
