/************************************************************************
 * © 2020-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.jdbc.hana;

import static com.sap.cds.DataStoreConfiguration.IGNORE_LOCALE_ON_HANA;
import static com.sap.cds.impl.sql.SQLHelper.commaSeparated;
import static java.util.stream.Collectors.joining;

import java.util.Locale;
import java.util.Optional;
import java.util.stream.Stream;

import com.sap.cds.DataStoreConfiguration;
import com.sap.cds.impl.sql.SQLHelper;
import com.sap.cds.jdbc.spi.StatementResolver;
import com.sap.cds.ql.cqn.CqnLock.Mode;
import com.sap.cds.ql.cqn.CqnStatement;

public class HanaStatementResolver implements StatementResolver {

	private final DataStoreConfiguration dataStoreConfiguration;
	private final int majorVersion;

	public HanaStatementResolver(DataStoreConfiguration dataStoreConfiguration, int majorVersion) {
		this.dataStoreConfiguration = dataStoreConfiguration;
		this.majorVersion = majorVersion;
	}

	@Override
	public Optional<String> timeoutClause(int timeoutSeconds) {
		return Optional.of("WAIT " + timeoutSeconds);
	}

	@Override
	public Optional<String> collateClause(CqnStatement statement, Locale locale) {
		boolean ignoreLocaleOnHana = dataStoreConfiguration.getProperty(IGNORE_LOCALE_ON_HANA, false);
		if (locale != null && !ignoreLocaleOnHana && statementNeedsCollateClause(statement)) {
			return Optional.of("with parameters('LOCALE' = " + SQLHelper.literal(locale.getLanguage()) + ")");
		}
		return Optional.empty();
	}

	private boolean statementNeedsCollateClause(CqnStatement statement) {
		return (statement.isDelete() && statement.asDelete().where().isPresent())
				|| (statement.isUpdate() && statement.asUpdate().where().isPresent()) || statement.isSelect();
	}

	/**
	 * HANA Upsert: Updates rows in a table or inserts new rows.
	 * 
	 * https://help.sap.com/docs/HANA_SERVICE_CF/7c78579ce9b14a669c1f3295b0d8ca16/ea8b6773be584203bcd99da76844c5ed.html
	 */
	@Override
	public String upsert(String table, Stream<String> keyColumns, Stream<String> upsertColumns,
			Stream<String> upsertValues) {
		String columns = commaSeparated(upsertColumns);
		String values = commaSeparated(upsertValues);

		return Stream.of("UPSERT", table, columns, "VALUES", values, "WITH PRIMARY KEY").collect(joining(" "));
	}

	@Override
	public Optional<String> lockClause(Mode mode) {
		final String clause;
		switch (mode) {
		case SHARED:
			if (majorVersion >= 4) {
				clause = "FOR SHARE LOCK";
				break;
			}
		default:
			clause = "FOR UPDATE";
		}
		return Optional.of(clause);
	}

}
