/*******************************************************************
 * © 2021 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.ql.impl;

import static com.sap.cds.impl.builder.model.Conjunction.and;
import static com.sap.cds.impl.builder.model.ElementRefImpl.element;
import static com.sap.cds.impl.builder.model.StructuredTypeRefImpl.typeRef;
import static com.sap.cds.impl.parser.token.RefSegmentImpl.copy;
import static com.sap.cds.ql.CQL.not;
import static com.sap.cds.ql.cqn.CqnMatchPredicate.Quantifier.ALL;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.joining;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.StringJoiner;

import com.sap.cds.impl.AssociationAnalyzer;
import com.sap.cds.impl.builder.model.ExistsSubquery;
import com.sap.cds.impl.builder.model.MatchPredicate;
import com.sap.cds.impl.builder.model.StructuredTypeRefImpl;
import com.sap.cds.impl.parser.token.RefSegmentImpl;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.RefSegment;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.Value;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnMatchPredicate.Quantifier;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnValidationException;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.ql.cqn.Modifier;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsElementNotFoundException;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.util.CdsModelUtils;

public class MatchPredicateNormalizer {

	private static final RefSegment $OUTER = RefSegmentImpl.refSegment(CqnExistsSubquery.OUTER);
	private final AssociationAnalyzer associationAnalyzer = new AssociationAnalyzer();
	private final CdsModel model;
	private final CdsStructuredType target;

	public MatchPredicateNormalizer(CdsModel model, CdsStructuredType target) {
		this.model = model;
		this.target = target;
	}

	public <S extends CqnStatement> S normalize(S s) {
		s = CQL.copy(s, new Path2Nested());
		s = CQL.copy(s, new Match2Exists());

		return s;
	}

	public Predicate normalize(CqnPredicate pred) {
		Predicate copy = CQL.copy(pred, new Path2Nested());
		copy = CQL.copy(copy, new Match2Exists());

		return copy;
	}

	private class Path2Nested implements Modifier {

		@Override
		public Predicate match(StructuredTypeRef ref, Predicate pred, Quantifier quantifier) {
			return convertToNestedIfNecessary(target, ref, pred, quantifier);
		}

		private Predicate convertToNestedIfNecessary(CdsStructuredType rowType, StructuredTypeRef ref, Predicate pred,
				Quantifier quantifier) {
			List<RefSegment> segments = ref.segments();

			int n = segments.size();
			for (int i = 1; i < n; i++) {
				Segment segment = segments.get(i - 1);
				CdsElement element = rowType.getElement(segment.id());
				CdsType elementType = element.getType();
				if (elementType.isStructured()) {
					rowType = elementType.as(CdsStructuredType.class);
				} else if (elementType.isAssociation()) {
					rowType = elementType.as(CdsAssociationType.class).getTarget();
					if (!CdsModelUtils.isSingleValued(elementType)) {
						// 1:n in the middle
						// author.books.publisher.journals[title='xyz']
						// 0: author, 1: books
						StructuredTypeRef prefix = StructuredTypeRefImpl.typeRef(segments.subList(0, i));
						StructuredTypeRef suffix = StructuredTypeRefImpl.typeRef(segments.subList(i, n));
						Predicate nested = convertToNestedIfNecessary(rowType, suffix, pred, quantifier);
						return MatchPredicate.match(prefix, quantifier, nested);
					}
				}
			}

			return MatchPredicate.match(ref, quantifier, pred);
		}

	}

	private class Match2Exists implements Modifier {

		@Override
		public CqnStructuredTypeRef ref(StructuredTypeRef ref) {
			Iterator<RefSegment> iter = ref.segments().iterator();

			RefSegment seg = iter.next();
			CdsStructuredType rowType = model.getStructuredType(seg.id());
			normalizeSegment(seg, rowType);

			while (iter.hasNext()) {
				seg = iter.next();
				rowType = navigate(rowType, seg.id());
				normalizeSegment(seg, rowType);
			}

			return ref;
		}

		@Override
		public Predicate match(StructuredTypeRef ref, Predicate pred, Quantifier quantifier) {
			CdsElement association = getAssociation(ref);
			CdsEntity target = association.getType().as(CdsAssociationType.class).getTarget();

			StructuredTypeRef from = absoluteRef(target, ref);
			rejectToManyPathInFilter(target, pred);

			CqnPredicate where = innerToOuter(getPrefix(ref), association);

			if (pred != null) {
				pred = normalize(target, pred);
				if (quantifier == ALL) {
					pred = not(pred);
				}
				where = and(where, pred);
			}

			CqnSelect subquery = Select.from(from).where(where);

			ExistsSubquery exists = new ExistsSubquery(subquery);

			if (quantifier == ALL) {
				return exists.not();
			}
			return exists;
		}

		private void rejectToManyPathInFilter(CdsEntity entity, Predicate pred) {
			if (pred == null) {
				return;
			}
			CqnVisitor visitor = new CqnVisitor() {
				@Override
				public void visit(CqnElementRef ref) {
					CdsStructuredType structType = entity;
					for (Segment segment : ref.segments()) {
						CdsType elType = structType.getElement(segment.id()).getType();
						if (elType.isStructured()) {
							structType = elType.as(CdsStructuredType.class);
						} else if (elType.isAssociation()) {
							CdsAssociationType assoc = elType.as(CdsAssociationType.class);
							structType = assoc.getTarget();
							if (CdsModelUtils.isToMany(assoc)) {
								String message = MessageFormat.format(
										"Association ''{0}'' with to-many target cardinality in anyMatch/allMatch is not supported",
										segment.id());
								throw new CqnValidationException(message);
							}
						}
					}
				}
			};
			pred.accept(visitor);
		}

		private CqnPredicate innerToOuter(List<Segment> outerPrefix, CdsElement association) {
			CqnPredicate on = associationAnalyzer.getOnCondition(association);
			Modifier m = new Modifier() {
				@Override
				public Value<?> ref(ElementRef<?> ref) {
					List<Segment> segments = new LinkedList<>(ref.segments());
					if (ref.firstSegment().equals(association.getName())) {
						segments.remove(0);
					} else {
						segments.addAll(0, outerPrefix);
					}

					return element(segments);
				}
			};
			on = ExpressionVisitor.copy(on, m);

			return on;
		}

		private StructuredTypeRef absoluteRef(CdsEntity target, CqnStructuredTypeRef ref) {
			RefSegment seg = copy(ref.targetSegment()).id(target.getQualifiedName());

			return typeRef(asList(seg));
		}

		private void normalizeSegment(RefSegment seg, CdsStructuredType rowType) {
			seg.filter().ifPresent(f -> seg.filter(normalize(rowType, f)));
		}

		private Predicate normalize(CdsStructuredType rowType, CqnPredicate pred) {
			return new MatchPredicateNormalizer(model, rowType).normalize(pred);
		}

		private List<Segment> getPrefix(CqnStructuredTypeRef ref) {
			List<? extends Segment> all = ref.segments();

			int size = all.size();
			List<? extends Segment> start = all.subList(0, size - 1);

			List<Segment> prefix = new ArrayList<>(size);
			prefix.add($OUTER);
			prefix.addAll(start);

			return prefix;
		}

		private CdsElement getAssociation(CqnStructuredTypeRef ref) {
			CdsStructuredType rowType = target;
			String root = rowType.getQualifiedName();
			CdsElement element = CdsModelUtils.element(rowType, ref.segments());
			CdsType elementType = element.getType();
			if (!elementType.isAssociation()) {
				String path = ref.segments().stream().map(Segment::id).collect(joining("."));
				String message = MessageFormat.format(
						"The reference {0}.{1} does not terminate in an association and can''t be used with the anyMatch predicate.",
						root, path);
				throw new CqnValidationException(message);
			}
			return element;
		}

	}

	private static CdsStructuredType navigate(CdsStructuredType rowType, String id) {
		try {
			CdsType type = rowType.getElement(id).getType();
			if (type.isStructured()) {
				return type.as(CdsStructuredType.class);
			}

			if (type.isAssociation()) {
				return type.as(CdsAssociationType.class).getTarget();
			}
		} catch (CdsElementNotFoundException e) {
			throw e;
		}

		String message = MessageFormat.format(
				"The reference {0}.{1} does not terminate in an association and can''t be used with the anyMatch predicate.",
				rowType.getQualifiedName(), id);
		throw new CqnValidationException(message);
	}

}
