/*******************************************************************
 * © 2021 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.impl;

import static com.sap.cds.Cds4jServiceLoader.loadAll;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.Collection;
import java.util.function.Supplier;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.CdsCommunicationException;
import com.sap.cds.CdsDataStoreException;
import com.sap.cds.DataStoreConfiguration;
import com.sap.cds.jdbc.spi.DbContext;
import com.sap.cds.jdbc.spi.DbContextFactory;

/**
 * Creates {@link DbContext} instances suitable for the DB behind the given
 * {@link Connection}.
 */
public class DbContextFactoryRegister {

	private DbContextFactoryRegister() {
		// empty
	}

	public static DbContext context(DataStoreConfiguration dataStoreConfiguration, Supplier<Connection> ds) {
		try (Connection conn = ds.get()) {
			DatabaseMetaData metaData = conn.getMetaData();
			String productName = metaData.getDatabaseProductName();
			int majorVersion = metaData.getDatabaseMajorVersion();

			return context(dataStoreConfiguration, productName, majorVersion);
		} catch (SQLException e) {
			throw new CdsCommunicationException("Cannot obtain connection to DB", e);
		}
	}

	@VisibleForTesting
	static DbContext context(DataStoreConfiguration dataStoreConfiguration, String productName, int majorVersion) {
		return SingletonHolder.instances.stream().filter(f -> f.supports(productName))
				.map(f -> f.create(dataStoreConfiguration, majorVersion)).findFirst()
				.orElseThrow(() -> new CdsDataStoreException("Unsupported DB " + productName));
	}
	
    private static class SingletonHolder {    
        private static final Collection<DbContextFactory> instances = loadAll(DbContextFactory.class);
    }    

}
