/*******************************************************************
 * © 2021 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.impl.docstore;

import java.util.Collections;

import com.sap.cds.SessionContext;
import com.sap.cds.impl.Context;
import com.sap.cds.impl.ContextImpl;
import com.sap.cds.impl.PreparedCqnStmt;
import com.sap.cds.impl.sql.SQLStatementBuilder;
import com.sap.cds.jdbc.spi.DbContext;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.util.CdsModelUtils;

public class DocStoreUtils {

	private final CdsModel model;

	public DocStoreUtils(CdsModel model) {
		this.model = model;
	}

	static String hanaDocStoreTypeFromCdsBaseType(CdsBaseType cdsBaseType) {
		switch (cdsBaseType) {
			case INTEGER:
			case INTEGER64:
				return "BIGINT";
			case DECIMAL:
			case DOUBLE:
				return "DOUBLE";
			case BOOLEAN:
				return "BOOLEAN";
			default:
				return "NVARCHAR";
		}
	}

	static String valueToParamCastExpression(PreparedCqnStmt.Parameter param) {
		return "CAST(? AS %s)".formatted(hanaDocStoreTypeFromCdsBaseType(param.type()));
	}

	public static SQLStatementBuilder.SQLStatement getDocStoreInsertStatement(CqnInsert insert, Context context) {
		CdsEntity entity = CdsModelUtils.entity(context.getCdsModel(), insert.ref());
		String sql = "INSERT INTO %s VALUES(?)".formatted(context.getTableNameResolver().tableName(entity));
		return new SQLStatementBuilder.SQLStatement(sql, Collections.singletonList(new PreparedCqnStmt.JsonParam()));
	}

	public boolean targetsDocStore(CqnStatement statement) {
		if (statement.isSelect()) {
			return targetsDocStore(statement.asSelect());
		}

		return targetsDocStore(statement.ref());
	}

	private boolean targetsDocStore(CqnStructuredTypeRef ref) {
		CdsEntity entity = CdsModelUtils.entity(model, ref);

		return targetsDocStore(entity);
	}

	boolean targetsDocStore(CqnSelect query) {
		if (!query.from().isRef()) {
			return false;
		}

		return targetsDocStore(query.ref());
	}

	public static boolean targetsDocStore(CdsStructuredType entity) {
		if (entity == null) {
			return false;
		}
		return entity.getAnnotationValue("cds.persistence.docstore", false);
	}

	public static Context wrap(Context ctx) {
		return wrap(ctx.getCdsModel(), ctx.getSessionContext(), ctx.getDbContext());
	}

	public static Context wrap(CdsModel model, SessionContext session, DbContext columnStoreContext) {
		DbContext docStoredContext = columnStoreContext.subDialect("JsonDocStore");
		return ContextImpl.context(model, docStoredContext, session);
	}
}
