/************************************************************************
 * © 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.impl.sql;

import static com.sap.cds.util.CdsModelUtils.element;

import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Stream;

import com.sap.cds.impl.Context;
import com.sap.cds.impl.PreparedCqnStmt.Parameter;
import com.sap.cds.impl.localized.LocaleUtils;
import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;

public interface SQLStatementBuilder {

	public static class SQLStatement {
		private final String sql;
		private final List<Parameter> params;

		public SQLStatement(String sql, List<Parameter> params) {
			this.sql = sql;
			this.params = params;
		}

		public String sql() {
			return sql;
		}

		public List<Parameter> params() {
			return params;
		}
	}

	SQLStatement build();

	public static <T> Stream<String> commaSeparated(Stream<T> stream, Function<T, String> f) {
		return stream.map(f).filter(t -> t != null).flatMap(e -> Stream.of(",", e)).skip(1);
	}

	static Stream<String> commaSeparated(Stream<String> stream) {
		return commaSeparated(stream, v -> v);
	}

	static Function<CqnElementRef, String> aliasResolver(Context context, CdsEntity entity) {
		SqlMapping sqlMapping = context.getDbContext().getSqlMapping(entity);

		Optional<String> collate = context.getDbContext().getStatementResolver()
				.collate(context.getSessionContext().getLocale());
		if (collate.isPresent()) {
			LocaleUtils localeUtils = new LocaleUtils(context.getCdsModel(), context.getDataStoreConfiguration());
			String collateClause = " " + collate.get();
			return ref -> {
				String column = sqlMapping.columnName(ref);
				CdsElement element = element(entity, ref);
				if (localeUtils.requiresCollate(element)) {
					column = column + collateClause;
				}
				return column;
			};
		}
		return sqlMapping::columnName;
	}
}
