/************************************************************************
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.jdbc.generic;

import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

import com.sap.cds.CdsDataStoreConnector;
import com.sap.cds.CdsDataStoreConnector.Capabilities;
import com.sap.cds.DataStoreConfiguration;
import com.sap.cds.impl.sql.SqlMappingImpl;
import com.sap.cds.jdbc.spi.DbContext;
import com.sap.cds.jdbc.spi.ExceptionAnalyzer;
import com.sap.cds.jdbc.spi.FunctionMapper;
import com.sap.cds.jdbc.spi.ScalarValueResolver;
import com.sap.cds.jdbc.spi.PredicateMapper;
import com.sap.cds.jdbc.spi.SearchResolver;
import com.sap.cds.jdbc.spi.SessionVariableSetter;
import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.jdbc.spi.StatementResolver;
import com.sap.cds.jdbc.spi.TableNameResolver;
import com.sap.cds.jdbc.spi.ValueBinder;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;

public abstract class GenericDbContext implements DbContext {

	protected final DataStoreConfiguration config;

	protected GenericDbContext(DataStoreConfiguration dataStoreConfiguration) {
		this.config = dataStoreConfiguration;
	}

	@Override
	public TableNameResolver createTableNameResolver(Locale locale) {
		return new LocaleVariableTableNameResolver(this::getSqlMapping, locale);
	}

	@Override
	public SearchResolver createSearchResolver(CdsModel model, Locale locale, Map<String, Object> hints) {
		return new GenericSearchResolver(config, model, locale);
	}

	@Override
	public FunctionMapper getFunctionMapper() {
		return new GenericFunctionMapper();
	}

	@Override
	public StatementResolver getStatementResolver() {
		return new GenericStatementResolver();
	}

	@Override
	public ExceptionAnalyzer getExceptionAnalyzer() {
		return new GenericExceptionAnalyzer();
	}

	@Override
	public Capabilities getCapabilities() {
		return CdsDataStoreConnector.Capabilities.ALL;
	}

	@Override
	public PredicateMapper getPredicateMapper() {
		return new GenericPredicateMapper();
	}

	@Override
	public ValueBinder getBinder(TimeZone timeZone) {
		return new GenericBinder(getCapabilities().timestampPrecision(), timeZone);
	}

	@Override
	public SessionVariableSetter getSessionVariableSetter() {
		return new GenericSessionVariableSetter();
	}

	@Override
	public SqlMapping getSqlMapping(CdsStructuredType rowType) {
		return new SqlMappingImpl(rowType, casing());
	}

	@Override
	public ScalarValueResolver getScalarValueResolver() {
		return new GenericScalarValueResolver();
	}

}
